/*
 * Decompiled with CFR 0.152.
 */
package com.example.vibe.ui.chat;

import com.example.vibe.core.model.LlmMessage;
import com.example.vibe.ui.chat.ChatMessage;
import com.example.vibe.ui.chat.MessageKind;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.UUID;
import java.util.concurrent.CopyOnWriteArrayList;
import java.util.function.Consumer;

public class ChatSession {
    private final String sessionId;
    private final List<ChatMessage> messages;
    private final List<SessionListener> listeners;
    private AgentRunState agentState;

    public ChatSession() {
        this(UUID.randomUUID().toString());
    }

    public ChatSession(String sessionId) {
        this.sessionId = Objects.requireNonNull(sessionId);
        this.messages = new ArrayList<ChatMessage>();
        this.listeners = new CopyOnWriteArrayList<SessionListener>();
        this.agentState = null;
    }

    public String getSessionId() {
        return this.sessionId;
    }

    public List<ChatMessage> getMessages() {
        return Collections.unmodifiableList(this.messages);
    }

    public int getMessageCount() {
        return this.messages.size();
    }

    public boolean isEmpty() {
        return this.messages.isEmpty();
    }

    public void addMessage(ChatMessage message) {
        Objects.requireNonNull(message, "message must not be null");
        this.messages.add(message);
        this.fireMessageAdded(message);
    }

    public ChatMessage addUserMessage(String content) {
        ChatMessage message = ChatMessage.user(content);
        this.addMessage(message);
        return message;
    }

    public ChatMessage addAssistantMessage(String content) {
        ChatMessage message = ChatMessage.assistant(content);
        this.addMessage(message);
        return message;
    }

    public ChatMessage addPendingAssistantMessage() {
        ChatMessage message = ChatMessage.pendingAssistant();
        this.addMessage(message);
        return message;
    }

    public ChatMessage addSystemMessage(String content) {
        ChatMessage message = ChatMessage.system(content);
        this.addMessage(message);
        return message;
    }

    public Optional<ChatMessage> findMessage(String messageId) {
        return this.messages.stream().filter(m -> m.getId().equals(messageId)).findFirst();
    }

    public Optional<ChatMessage> getLastMessage(MessageKind kind) {
        int i = this.messages.size() - 1;
        while (i >= 0) {
            ChatMessage message = this.messages.get(i);
            if (message.getKind() == kind && !message.isSuperseded()) {
                return Optional.of(message);
            }
            --i;
        }
        return Optional.empty();
    }

    public Optional<ChatMessage> getLastAssistantMessage() {
        return this.getLastMessage(MessageKind.ASSISTANT);
    }

    public Optional<ChatMessage> getLastUserMessage() {
        return this.getLastMessage(MessageKind.USER);
    }

    public boolean editMessage(String messageId, String newContent) {
        int index = -1;
        int i = 0;
        while (i < this.messages.size()) {
            if (this.messages.get(i).getId().equals(messageId)) {
                index = i;
                break;
            }
            ++i;
        }
        if (index < 0) {
            return false;
        }
        ChatMessage message = this.messages.get(index);
        if (!message.getKind().isEditable()) {
            return false;
        }
        ArrayList<ChatMessage> removed = new ArrayList<ChatMessage>();
        while (this.messages.size() > index + 1) {
            removed.add(this.messages.remove(this.messages.size() - 1));
        }
        ChatMessage newMessage = ChatMessage.user(newContent);
        this.messages.set(index, newMessage);
        this.fireMessageEdited(message, newMessage, removed);
        return true;
    }

    public Optional<ChatMessage> regenerateLastResponse() {
        Optional<ChatMessage> lastAssistant = this.getLastAssistantMessage();
        if (lastAssistant.isEmpty()) {
            return Optional.empty();
        }
        ChatMessage original = lastAssistant.get();
        ChatMessage newMessage = ChatMessage.pendingAssistant();
        newMessage.markRegeneratedFrom(original.getId());
        original.markSupersededBy(newMessage.getId());
        this.addMessage(newMessage);
        this.fireMessageRegenerated(original, newMessage);
        return Optional.of(newMessage);
    }

    public List<LlmMessage> toConversationHistory() {
        ArrayList<LlmMessage> history = new ArrayList<LlmMessage>();
        for (ChatMessage message : this.messages) {
            if (message.getKind() == MessageKind.SYSTEM || message.isSuperseded() || message.isPending()) continue;
            history.add(message.toLlmMessage());
        }
        return history;
    }

    public void clear() {
        ArrayList<ChatMessage> removed = new ArrayList<ChatMessage>(this.messages);
        this.messages.clear();
        this.agentState = null;
        this.fireSessionCleared(removed);
    }

    public AgentRunState getAgentState() {
        return this.agentState;
    }

    public AgentRunState startAgentRun() {
        this.agentState = new AgentRunState();
        this.fireAgentStateChanged(this.agentState);
        return this.agentState;
    }

    public void endAgentRun() {
        if (this.agentState != null) {
            this.agentState.complete();
            this.fireAgentStateChanged(this.agentState);
            this.agentState = null;
        }
    }

    public boolean isAgentRunning() {
        return this.agentState != null && this.agentState.isRunning();
    }

    public void addListener(SessionListener listener) {
        if (listener != null) {
            this.listeners.add(listener);
        }
    }

    public void removeListener(SessionListener listener) {
        this.listeners.remove(listener);
    }

    private void fireMessageAdded(ChatMessage message) {
        this.forEachListener(l -> l.onMessageAdded(this, message));
    }

    private void fireMessageEdited(ChatMessage original, ChatMessage newMessage, List<ChatMessage> removed) {
        this.forEachListener(l -> l.onMessageEdited(this, original, newMessage, removed));
    }

    private void fireMessageRegenerated(ChatMessage original, ChatMessage newMessage) {
        this.forEachListener(l -> l.onMessageRegenerated(this, original, newMessage));
    }

    private void fireSessionCleared(List<ChatMessage> removed) {
        this.forEachListener(l -> l.onSessionCleared(this, removed));
    }

    private void fireAgentStateChanged(AgentRunState state) {
        this.forEachListener(l -> l.onAgentStateChanged(this, state));
    }

    private void forEachListener(Consumer<SessionListener> action) {
        for (SessionListener listener : this.listeners) {
            try {
                action.accept(listener);
            }
            catch (Exception exception) {}
        }
    }

    public static class AgentRunState {
        private final String runId;
        private int currentIteration = 0;
        private int maxIterations = 10;
        private boolean running = true;
        private boolean cancelled = false;
        private final List<ToolExecutionStep> steps = new ArrayList<ToolExecutionStep>();

        public AgentRunState() {
            this.runId = UUID.randomUUID().toString();
        }

        public String getRunId() {
            return this.runId;
        }

        public int getCurrentIteration() {
            return this.currentIteration;
        }

        public int getMaxIterations() {
            return this.maxIterations;
        }

        public boolean isRunning() {
            return this.running && !this.cancelled;
        }

        public boolean isCancelled() {
            return this.cancelled;
        }

        public List<ToolExecutionStep> getSteps() {
            return Collections.unmodifiableList(this.steps);
        }

        public void incrementIteration() {
            ++this.currentIteration;
        }

        public void addStep(ToolExecutionStep step) {
            this.steps.add(step);
        }

        public void cancel() {
            this.cancelled = true;
            this.running = false;
        }

        public void complete() {
            this.running = false;
        }

        public static class ToolExecutionStep {
            private final String toolCallId;
            private final String toolName;
            private StepStatus status;
            private String result;
            private String error;

            public ToolExecutionStep(String toolCallId, String toolName) {
                this.toolCallId = toolCallId;
                this.toolName = toolName;
                this.status = StepStatus.PENDING;
            }

            public String getToolCallId() {
                return this.toolCallId;
            }

            public String getToolName() {
                return this.toolName;
            }

            public StepStatus getStatus() {
                return this.status;
            }

            public String getResult() {
                return this.result;
            }

            public String getError() {
                return this.error;
            }

            public void markRunning() {
                this.status = StepStatus.RUNNING;
            }

            public void markSuccess(String result) {
                this.status = StepStatus.SUCCESS;
                this.result = result;
            }

            public void markFailed(String error) {
                this.status = StepStatus.FAILED;
                this.error = error;
            }

            public void markCancelled() {
                this.status = StepStatus.CANCELLED;
            }

            public static enum StepStatus {
                PENDING,
                RUNNING,
                SUCCESS,
                FAILED,
                CANCELLED;

            }
        }
    }

    public static interface SessionListener {
        default public void onMessageAdded(ChatSession session, ChatMessage message) {
        }

        default public void onMessageEdited(ChatSession session, ChatMessage original, ChatMessage newMessage, List<ChatMessage> removed) {
        }

        default public void onMessageRegenerated(ChatSession session, ChatMessage original, ChatMessage newMessage) {
        }

        default public void onSessionCleared(ChatSession session, List<ChatMessage> removed) {
        }

        default public void onAgentStateChanged(ChatSession session, AgentRunState state) {
        }
    }
}

