/*
 * Decompiled with CFR 0.152.
 */
package com.example.vibe.core.tools;

import com.example.vibe.core.logging.LogSanitizer;
import com.example.vibe.core.logging.VibeLogger;
import com.example.vibe.core.tools.ITool;
import com.example.vibe.core.tools.ToolResult;
import java.io.BufferedReader;
import java.io.File;
import java.io.IOException;
import java.io.InputStreamReader;
import java.nio.charset.Charset;
import java.nio.charset.StandardCharsets;
import java.util.Arrays;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.TimeUnit;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IWorkspaceRoot;
import org.eclipse.core.resources.ResourcesPlugin;

public class ShellTool
implements ITool {
    private static final VibeLogger.CategoryLogger LOG = VibeLogger.forClass(ShellTool.class);
    private static final String SCHEMA = "{\n    \"type\": \"object\",\n    \"properties\": {\n        \"command\": {\n            \"type\": \"string\",\n            \"description\": \"Shell command to execute\"\n        },\n        \"working_directory\": {\n            \"type\": \"string\",\n            \"description\": \"Working directory (workspace-relative path or project name, optional)\"\n        },\n        \"timeout_seconds\": {\n            \"type\": \"integer\",\n            \"description\": \"Timeout in seconds (default: 60, max: 300)\"\n        }\n    },\n    \"required\": [\"command\"]\n}\n";
    private static final int DEFAULT_TIMEOUT_SECONDS = 60;
    private static final int MAX_TIMEOUT_SECONDS = 300;
    private static final int MAX_OUTPUT_CHARS = 50000;
    private static final boolean IS_WINDOWS = System.getProperty("os.name").toLowerCase().contains("win");

    @Override
    public String getName() {
        return "shell";
    }

    @Override
    public String getDescription() {
        return "\u0412\u044b\u043f\u043e\u043b\u043d\u044f\u0435\u0442 shell \u043a\u043e\u043c\u0430\u043d\u0434\u0443. \u041d\u0430 Windows \u0438\u0441\u043f\u043e\u043b\u044c\u0437\u0443\u0435\u0442 PowerShell \u0438\u043b\u0438 cmd, \u043d\u0430 Unix - bash. \u0420\u0430\u0431\u043e\u0447\u0430\u044f \u0434\u0438\u0440\u0435\u043a\u0442\u043e\u0440\u0438\u044f \u043f\u043e \u0443\u043c\u043e\u043b\u0447\u0430\u043d\u0438\u044e - \u043a\u043e\u0440\u0435\u043d\u044c workspace.";
    }

    @Override
    public String getParameterSchema() {
        return SCHEMA;
    }

    @Override
    public boolean requiresConfirmation() {
        return true;
    }

    @Override
    public boolean isDestructive() {
        return true;
    }

    @Override
    public CompletableFuture<ToolResult> execute(Map<String, Object> parameters) {
        return CompletableFuture.supplyAsync(() -> {
            long startTime = System.currentTimeMillis();
            String command = (String)parameters.get("command");
            if (command == null || command.isEmpty()) {
                LOG.warn("shell: \u043e\u0442\u0441\u0443\u0442\u0441\u0442\u0432\u0443\u0435\u0442 \u043f\u0430\u0440\u0430\u043c\u0435\u0442\u0440 command");
                return ToolResult.failure("\u041f\u0430\u0440\u0430\u043c\u0435\u0442\u0440 command \u043e\u0431\u044f\u0437\u0430\u0442\u0435\u043b\u0435\u043d");
            }
            String dangerCheck = this.checkDangerousCommand(command);
            if (dangerCheck != null) {
                LOG.warn("shell: \u043a\u043e\u043c\u0430\u043d\u0434\u0430 \u0437\u0430\u0431\u043b\u043e\u043a\u0438\u0440\u043e\u0432\u0430\u043d\u0430: %s (pattern: %s)", LogSanitizer.truncate(command, 100), dangerCheck);
                return ToolResult.failure("\u041a\u043e\u043c\u0430\u043d\u0434\u0430 \u0437\u0430\u0431\u043b\u043e\u043a\u0438\u0440\u043e\u0432\u0430\u043d\u0430: " + dangerCheck);
            }
            int timeoutSeconds = 60;
            Object timeoutParam = parameters.get("timeout_seconds");
            if (timeoutParam instanceof Number) {
                timeoutSeconds = Math.min(((Number)timeoutParam).intValue(), 300);
            }
            File workingDir = this.resolveWorkingDirectory((String)parameters.get("working_directory"));
            LOG.info("shell: \u0432\u044b\u043f\u043e\u043b\u043d\u0435\u043d\u0438\u0435 \u043a\u043e\u043c\u0430\u043d\u0434\u044b: %s (timeout=%d \u0441\u0435\u043a, cwd=%s)", LogSanitizer.truncate(command, 200), timeoutSeconds, workingDir.getName());
            try {
                ToolResult result = this.executeCommand(command, workingDir, timeoutSeconds);
                long duration = System.currentTimeMillis() - startTime;
                LOG.debug("shell: \u0437\u0430\u0432\u0435\u0440\u0448\u0435\u043d\u043e \u0437\u0430 %s, success=%b", LogSanitizer.formatDuration(duration), result.isSuccess());
                return result;
            }
            catch (Exception e) {
                LOG.error("shell: \u043e\u0448\u0438\u0431\u043a\u0430 \u0432\u044b\u043f\u043e\u043b\u043d\u0435\u043d\u0438\u044f \u043a\u043e\u043c\u0430\u043d\u0434\u044b: %s", e.getMessage());
                return ToolResult.failure("\u041e\u0448\u0438\u0431\u043a\u0430 \u0432\u044b\u043f\u043e\u043b\u043d\u0435\u043d\u0438\u044f: " + e.getMessage());
            }
        });
    }

    private ToolResult executeCommand(String command, File workingDir, int timeoutSeconds) throws IOException, InterruptedException {
        Object[] shellCommand = this.getShellCommand(command);
        LOG.debug("shell: shellCommand=%s, workingDir=%s", Arrays.toString(shellCommand), workingDir.getAbsolutePath());
        ProcessBuilder pb = new ProcessBuilder((String[])shellCommand);
        pb.directory(workingDir);
        pb.redirectErrorStream(true);
        if (IS_WINDOWS) {
            pb.environment().put("PYTHONIOENCODING", "utf-8");
            pb.environment().put("PYTHONUTF8", "1");
        }
        long startTime = System.currentTimeMillis();
        Process process = pb.start();
        StringBuilder output = new StringBuilder();
        Thread outputReader = new Thread(() -> {
            try {
                Throwable throwable = null;
                Object var4_5 = null;
                try (BufferedReader reader = new BufferedReader(new InputStreamReader(process.getInputStream(), this.getConsoleCharset()));){
                    String line;
                    while ((line = reader.readLine()) != null) {
                        if (output.length() >= 50000) continue;
                        output.append(line).append("\n");
                    }
                }
                catch (Throwable throwable2) {
                    if (throwable == null) {
                        throwable = throwable2;
                    } else if (throwable != throwable2) {
                        throwable.addSuppressed(throwable2);
                    }
                    throw throwable;
                }
            }
            catch (IOException iOException) {}
        });
        outputReader.start();
        boolean completed = process.waitFor(timeoutSeconds, TimeUnit.SECONDS);
        long executionTime = System.currentTimeMillis() - startTime;
        if (!completed) {
            this.killProcessTree(process);
            return ToolResult.failure(String.format("\u041a\u043e\u043c\u0430\u043d\u0434\u0430 \u043f\u0440\u0435\u0432\u044b\u0441\u0438\u043b\u0430 \u0442\u0430\u0439\u043c\u0430\u0443\u0442 (%d \u0441\u0435\u043a). \u0427\u0430\u0441\u0442\u0438\u0447\u043d\u044b\u0439 \u0432\u044b\u0432\u043e\u0434:\n%s", timeoutSeconds, this.truncateOutput(output.toString())));
        }
        outputReader.join(1000L);
        int exitCode = process.exitValue();
        StringBuilder result = new StringBuilder();
        result.append("**\u041a\u043e\u043c\u0430\u043d\u0434\u0430:** `").append(command).append("`\n");
        result.append("**\u0420\u0430\u0431\u043e\u0447\u0430\u044f \u0434\u0438\u0440\u0435\u043a\u0442\u043e\u0440\u0438\u044f:** `").append(workingDir.getAbsolutePath()).append("`\n");
        result.append("**\u041a\u043e\u0434 \u0437\u0430\u0432\u0435\u0440\u0448\u0435\u043d\u0438\u044f:** ").append(exitCode).append("\n");
        result.append("**\u0412\u0440\u0435\u043c\u044f \u0432\u044b\u043f\u043e\u043b\u043d\u0435\u043d\u0438\u044f:** ").append(executionTime).append(" \u043c\u0441\n\n");
        String outputStr = this.truncateOutput(output.toString());
        if (!outputStr.isEmpty()) {
            result.append("**\u0412\u044b\u0432\u043e\u0434:**\n```\n").append(outputStr).append("\n```");
        } else {
            result.append("*(\u043d\u0435\u0442 \u0432\u044b\u0432\u043e\u0434\u0430)*");
        }
        if (exitCode == 0) {
            return ToolResult.success(result.toString(), ToolResult.ToolResultType.TEXT);
        }
        return ToolResult.failure(result.toString());
    }

    private String[] getShellCommand(String command) {
        if (IS_WINDOWS) {
            if (this.isPowerShellAvailable()) {
                return new String[]{"powershell.exe", "-NoProfile", "-NonInteractive", "-ExecutionPolicy", "Bypass", "-Command", command};
            }
            return new String[]{"cmd.exe", "/c", command};
        }
        String shell = System.getenv("SHELL");
        if (shell == null || shell.isEmpty()) {
            shell = "/bin/bash";
        }
        return new String[]{shell, "-c", command};
    }

    private boolean isPowerShellAvailable() {
        try {
            Process p = new ProcessBuilder("powershell.exe", "-NoProfile", "-Command", "exit 0").redirectErrorStream(true).start();
            return p.waitFor(5L, TimeUnit.SECONDS) && p.exitValue() == 0;
        }
        catch (Exception exception) {
            return false;
        }
    }

    private Charset getConsoleCharset() {
        if (IS_WINDOWS) {
            String encoding = System.getProperty("sun.jnu.encoding");
            if (encoding != null) {
                try {
                    return Charset.forName(encoding);
                }
                catch (Exception exception) {}
            }
            try {
                return Charset.forName("windows-1251");
            }
            catch (Exception exception) {
                return StandardCharsets.UTF_8;
            }
        }
        return StandardCharsets.UTF_8;
    }

    private String checkDangerousCommand(String command) {
        String lower = command.toLowerCase();
        Set<String> dangerousPatterns = this.getDangerousPatterns();
        for (String pattern : dangerousPatterns) {
            if (!lower.contains(pattern)) continue;
            return pattern;
        }
        if (lower.contains(":(){ :|:&}") || lower.contains("fork")) {
            return "\u043f\u043e\u0442\u0435\u043d\u0446\u0438\u0430\u043b\u044c\u043d\u0430\u044f fork-\u0431\u043e\u043c\u0431\u0430";
        }
        return null;
    }

    private Set<String> getDangerousPatterns() {
        HashSet<String> patterns = new HashSet<String>(Arrays.asList("rm -rf /", "rm -rf /*", "> /dev/sda", "mkfs", "dd if=", "shutdown", "reboot", "halt", "init 0", "init 6"));
        if (IS_WINDOWS) {
            patterns.addAll(Arrays.asList("format c:", "rd /s /q c:", "del /f /s /q c:", "reg delete", "bcdedit"));
        }
        return patterns;
    }

    private File resolveWorkingDirectory(String path) {
        IWorkspaceRoot root = ResourcesPlugin.getWorkspace().getRoot();
        if (path == null || path.isEmpty()) {
            return new File(root.getLocation().toOSString());
        }
        IProject project = root.getProject(path);
        if (project.exists() && project.getLocation() != null) {
            return new File(project.getLocation().toOSString());
        }
        File resolved = new File(root.getLocation().toOSString(), path);
        if (resolved.exists() && resolved.isDirectory()) {
            return resolved;
        }
        return new File(root.getLocation().toOSString());
    }

    private String truncateOutput(String output) {
        if (output.length() > 50000) {
            return output.substring(0, 50000) + "\n\n... (\u0432\u044b\u0432\u043e\u0434 \u043e\u0431\u0440\u0435\u0437\u0430\u043d, \u043f\u0440\u0435\u0432\u044b\u0448\u0435\u043d \u043b\u0438\u043c\u0438\u0442 50000 \u0441\u0438\u043c\u0432\u043e\u043b\u043e\u0432)";
        }
        return output;
    }

    private void killProcessTree(Process process) {
        try {
            if (IS_WINDOWS) {
                long pid = process.pid();
                Runtime.getRuntime().exec("taskkill /F /T /PID " + pid);
            } else {
                long pid = process.pid();
                Runtime.getRuntime().exec("kill -9 -" + pid);
            }
        }
        catch (Exception exception) {
            process.destroyForcibly();
        }
    }
}

