/*
 * Decompiled with CFR 0.152.
 */
package com.example.vibe.core.session;

import com.example.vibe.core.internal.VibeCorePlugin;
import com.example.vibe.core.session.ISessionStore;
import com.example.vibe.core.session.Session;
import com.example.vibe.core.session.SessionMessage;
import com.google.gson.Gson;
import com.google.gson.GsonBuilder;
import com.google.gson.TypeAdapter;
import com.google.gson.stream.JsonReader;
import com.google.gson.stream.JsonWriter;
import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.IOException;
import java.io.Reader;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.attribute.FileAttribute;
import java.time.Duration;
import java.time.Instant;
import java.util.ArrayList;
import java.util.Comparator;
import java.util.List;
import java.util.Optional;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import org.eclipse.core.runtime.ILog;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Platform;
import org.eclipse.core.runtime.Status;

public class FileSessionStore
implements ISessionStore {
    private static final String PLUGIN_ID = "com.example.vibe.core";
    private static final ILog LOG = Platform.getLog(FileSessionStore.class);
    private static final String SESSIONS_DIR = "sessions";
    private static final String SESSION_EXTENSION = ".json";
    private final Path sessionsDirectory;
    private final Gson gson;

    public FileSessionStore() {
        this(FileSessionStore.getDefaultSessionsDirectory());
    }

    public FileSessionStore(Path sessionsDirectory) {
        this.sessionsDirectory = sessionsDirectory;
        this.gson = this.createGson();
        this.ensureDirectoryExists();
    }

    private static Path getDefaultSessionsDirectory() {
        VibeCorePlugin plugin = VibeCorePlugin.getDefault();
        if (plugin != null) {
            IPath stateLoc = plugin.getStateLocation();
            return Path.of(stateLoc.toOSString(), new String[0]).resolve(SESSIONS_DIR);
        }
        return Path.of(System.getProperty("user.home"), ".vibe-sessions");
    }

    private Gson createGson() {
        return new GsonBuilder().setPrettyPrinting().registerTypeAdapter(Instant.class, (Object)new InstantTypeAdapter()).create();
    }

    private void ensureDirectoryExists() {
        try {
            Files.createDirectories(this.sessionsDirectory, new FileAttribute[0]);
        }
        catch (IOException e) {
            this.logError("\u041d\u0435 \u0443\u0434\u0430\u043b\u043e\u0441\u044c \u0441\u043e\u0437\u0434\u0430\u0442\u044c \u0434\u0438\u0440\u0435\u043a\u0442\u043e\u0440\u0438\u044e \u0441\u0435\u0441\u0441\u0438\u0439: " + String.valueOf(this.sessionsDirectory), e);
        }
    }

    private Path getSessionFile(String sessionId) {
        String safeId = this.sanitizeFileName(sessionId);
        return this.sessionsDirectory.resolve(safeId + SESSION_EXTENSION);
    }

    private String sanitizeFileName(String name) {
        return name.replaceAll("[^a-zA-Z0-9_-]", "_");
    }

    @Override
    public void save(Session session) throws ISessionStore.SessionStoreException {
        if (session == null) {
            throw new ISessionStore.SessionStoreException("\u0421\u0435\u0441\u0441\u0438\u044f \u043d\u0435 \u043c\u043e\u0436\u0435\u0442 \u0431\u044b\u0442\u044c null");
        }
        Path file = this.getSessionFile(session.getId());
        try {
            Throwable throwable = null;
            Object var4_6 = null;
            try (BufferedWriter writer = Files.newBufferedWriter(file, StandardCharsets.UTF_8, new OpenOption[0]);){
                SessionData data = SessionData.from(session);
                this.gson.toJson((Object)data, (Appendable)writer);
                this.logInfo("\u0421\u0435\u0441\u0441\u0438\u044f \u0441\u043e\u0445\u0440\u0430\u043d\u0435\u043d\u0430: " + session.getId());
            }
            catch (Throwable throwable2) {
                if (throwable == null) {
                    throwable = throwable2;
                } else if (throwable != throwable2) {
                    throwable.addSuppressed(throwable2);
                }
                throw throwable;
            }
        }
        catch (IOException e) {
            throw new ISessionStore.SessionStoreException("\u041e\u0448\u0438\u0431\u043a\u0430 \u0441\u043e\u0445\u0440\u0430\u043d\u0435\u043d\u0438\u044f \u0441\u0435\u0441\u0441\u0438\u0438: " + session.getId(), e);
        }
    }

    @Override
    public Optional<Session> load(String sessionId) throws ISessionStore.SessionStoreException {
        if (sessionId == null || sessionId.isEmpty()) {
            return Optional.empty();
        }
        Path file = this.getSessionFile(sessionId);
        if (!Files.exists(file, new LinkOption[0])) {
            return Optional.empty();
        }
        try {
            Throwable throwable = null;
            Object var4_7 = null;
            try (BufferedReader reader = Files.newBufferedReader(file, StandardCharsets.UTF_8);){
                SessionData data = (SessionData)this.gson.fromJson((Reader)reader, SessionData.class);
                if (data == null) {
                    return Optional.empty();
                }
                Session session = data.toSession();
                this.logInfo("\u0421\u0435\u0441\u0441\u0438\u044f \u0437\u0430\u0433\u0440\u0443\u0436\u0435\u043d\u0430: " + sessionId);
                return Optional.of(session);
            }
            catch (Throwable throwable2) {
                if (throwable == null) {
                    throwable = throwable2;
                } else if (throwable != throwable2) {
                    throwable.addSuppressed(throwable2);
                }
                throw throwable;
            }
        }
        catch (IOException e) {
            throw new ISessionStore.SessionStoreException("\u041e\u0448\u0438\u0431\u043a\u0430 \u0437\u0430\u0433\u0440\u0443\u0437\u043a\u0438 \u0441\u0435\u0441\u0441\u0438\u0438: " + sessionId, e);
        }
        catch (Exception e) {
            this.logError("\u041e\u0448\u0438\u0431\u043a\u0430 \u043f\u0430\u0440\u0441\u0438\u043d\u0433\u0430 \u0441\u0435\u0441\u0441\u0438\u0438: " + sessionId, e);
            return Optional.empty();
        }
    }

    @Override
    public boolean delete(String sessionId) throws ISessionStore.SessionStoreException {
        if (sessionId == null || sessionId.isEmpty()) {
            return false;
        }
        Path file = this.getSessionFile(sessionId);
        try {
            boolean deleted = Files.deleteIfExists(file);
            if (deleted) {
                this.logInfo("\u0421\u0435\u0441\u0441\u0438\u044f \u0443\u0434\u0430\u043b\u0435\u043d\u0430: " + sessionId);
            }
            return deleted;
        }
        catch (IOException e) {
            throw new ISessionStore.SessionStoreException("\u041e\u0448\u0438\u0431\u043a\u0430 \u0443\u0434\u0430\u043b\u0435\u043d\u0438\u044f \u0441\u0435\u0441\u0441\u0438\u0438: " + sessionId, e);
        }
    }

    @Override
    public boolean exists(String sessionId) {
        if (sessionId == null || sessionId.isEmpty()) {
            return false;
        }
        return Files.exists(this.getSessionFile(sessionId), new LinkOption[0]);
    }

    @Override
    public List<ISessionStore.SessionSummary> listAll() throws ISessionStore.SessionStoreException {
        return this.listSessionFiles().stream().map(this::loadSummary).filter(Optional::isPresent).map(Optional::get).sorted(Comparator.comparing(ISessionStore.SessionSummary::getUpdatedAt).reversed()).collect(Collectors.toList());
    }

    @Override
    public List<ISessionStore.SessionSummary> listByProject(String projectPath) throws ISessionStore.SessionStoreException {
        if (projectPath == null || projectPath.isEmpty()) {
            return new ArrayList<ISessionStore.SessionSummary>();
        }
        return this.listAll().stream().filter(s -> projectPath.equals(this.getProjectPathForSession(s.getId()))).collect(Collectors.toList());
    }

    @Override
    public List<ISessionStore.SessionSummary> listRecent(int limit) throws ISessionStore.SessionStoreException {
        return this.listAll().stream().limit(Math.max(0, limit)).collect(Collectors.toList());
    }

    @Override
    public int purgeOldSessions(int maxAgeDays) throws ISessionStore.SessionStoreException {
        if (maxAgeDays <= 0) {
            return 0;
        }
        Instant cutoff = Instant.now().minus(Duration.ofDays(maxAgeDays));
        int deleted = 0;
        for (Path file : this.listSessionFiles()) {
            try {
                ISessionStore.SessionSummary s;
                Optional<ISessionStore.SessionSummary> summary = this.loadSummary(file);
                if (!summary.isPresent() || (s = summary.get()).getStatus() != Session.SessionStatus.ARCHIVED || !s.getUpdatedAt().isBefore(cutoff)) continue;
                Files.deleteIfExists(file);
                ++deleted;
            }
            catch (IOException e) {
                this.logWarning("\u041d\u0435 \u0443\u0434\u0430\u043b\u043e\u0441\u044c \u0443\u0434\u0430\u043b\u0438\u0442\u044c \u0441\u0442\u0430\u0440\u0443\u044e \u0441\u0435\u0441\u0441\u0438\u044e: " + String.valueOf(file), e);
            }
        }
        if (deleted > 0) {
            this.logInfo("\u0423\u0434\u0430\u043b\u0435\u043d\u043e \u0441\u0442\u0430\u0440\u044b\u0445 \u0441\u0435\u0441\u0441\u0438\u0439: " + deleted);
        }
        return deleted;
    }

    private List<Path> listSessionFiles() {
        try {
            Throwable throwable = null;
            Object var2_4 = null;
            try (Stream<Path> stream = Files.list(this.sessionsDirectory);){
                return stream.filter(p -> p.toString().endsWith(SESSION_EXTENSION)).filter(path -> Files.isRegularFile(path, new LinkOption[0])).collect(Collectors.toList());
            }
            catch (Throwable throwable2) {
                if (throwable == null) {
                    throwable = throwable2;
                } else if (throwable != throwable2) {
                    throwable.addSuppressed(throwable2);
                }
                throw throwable;
            }
        }
        catch (IOException e) {
            this.logError("\u041e\u0448\u0438\u0431\u043a\u0430 \u0447\u0442\u0435\u043d\u0438\u044f \u0434\u0438\u0440\u0435\u043a\u0442\u043e\u0440\u0438\u0438 \u0441\u0435\u0441\u0441\u0438\u0439", e);
            return new ArrayList<Path>();
        }
    }

    private Optional<ISessionStore.SessionSummary> loadSummary(Path file) {
        try {
            Throwable throwable = null;
            Object var3_5 = null;
            try (BufferedReader reader = Files.newBufferedReader(file, StandardCharsets.UTF_8);){
                SessionData data = (SessionData)this.gson.fromJson((Reader)reader, SessionData.class);
                if (data != null) {
                    return Optional.of(data.toSummary());
                }
            }
            catch (Throwable throwable2) {
                if (throwable == null) {
                    throwable = throwable2;
                } else if (throwable != throwable2) {
                    throwable.addSuppressed(throwable2);
                }
                throw throwable;
            }
        }
        catch (Exception e) {
            this.logWarning("\u041e\u0448\u0438\u0431\u043a\u0430 \u0447\u0442\u0435\u043d\u0438\u044f \u0441\u0435\u0441\u0441\u0438\u0438: " + String.valueOf(file), e);
        }
        return Optional.empty();
    }

    private String getProjectPathForSession(String sessionId) {
        try {
            Optional<Session> session = this.load(sessionId);
            return session.map(Session::getProjectPath).orElse(null);
        }
        catch (ISessionStore.SessionStoreException sessionStoreException) {
            return null;
        }
    }

    public Path getSessionsDirectory() {
        return this.sessionsDirectory;
    }

    private void logInfo(String message) {
        LOG.log((IStatus)new Status(1, PLUGIN_ID, message));
    }

    private void logWarning(String message, Throwable error) {
        LOG.log((IStatus)new Status(2, PLUGIN_ID, message, error));
    }

    private void logError(String message, Throwable error) {
        LOG.log((IStatus)new Status(4, PLUGIN_ID, message, error));
    }

    private static class InstantTypeAdapter
    extends TypeAdapter<Instant> {
        private InstantTypeAdapter() {
        }

        public void write(JsonWriter out, Instant value) throws IOException {
            if (value == null) {
                out.nullValue();
            } else {
                out.value(value.toString());
            }
        }

        public Instant read(JsonReader in) throws IOException {
            String value = in.nextString();
            if (value == null || value.isEmpty()) {
                return null;
            }
            return Instant.parse(value);
        }
    }

    private static class MessageData {
        String id;
        String type;
        String content;
        Instant timestamp;
        String toolCallId;
        String toolName;
        boolean isError;

        private MessageData() {
        }

        static MessageData from(SessionMessage msg) {
            MessageData data = new MessageData();
            data.id = msg.getId();
            data.type = msg.getType().name();
            data.content = msg.getContent();
            data.timestamp = msg.getTimestamp();
            data.toolCallId = msg.getToolCallId();
            data.toolName = msg.getToolName();
            data.isError = msg.isError();
            return data;
        }

        SessionMessage toMessage() {
            return SessionMessage.builder().id(this.id).type(SessionMessage.MessageType.valueOf(this.type)).content(this.content).timestamp(this.timestamp).toolCallId(this.toolCallId).toolName(this.toolName).isError(this.isError).build();
        }
    }

    private static class SessionData {
        String id;
        String title;
        String projectPath;
        String projectName;
        String status;
        Instant createdAt;
        Instant updatedAt;
        String systemPrompt;
        String agentProfile;
        int totalTokens;
        List<MessageData> messages;

        private SessionData() {
        }

        static SessionData from(Session session) {
            SessionData data = new SessionData();
            data.id = session.getId();
            data.title = session.getTitle();
            data.projectPath = session.getProjectPath();
            data.projectName = session.getProjectName();
            data.status = session.getStatus().name();
            data.createdAt = session.getCreatedAt();
            data.updatedAt = session.getUpdatedAt();
            data.systemPrompt = session.getSystemPrompt();
            data.agentProfile = session.getAgentProfile();
            data.totalTokens = session.getTotalTokens();
            data.messages = session.getMessages().stream().map(MessageData::from).collect(Collectors.toList());
            return data;
        }

        Session toSession() {
            Session session = new Session(this.id);
            session.setTitle(this.title);
            session.setProjectPath(this.projectPath);
            session.setProjectName(this.projectName);
            session.setStatus(Session.SessionStatus.valueOf(this.status));
            session.setUpdatedAt(this.updatedAt);
            session.setSystemPrompt(this.systemPrompt);
            session.setAgentProfile(this.agentProfile);
            session.setTotalTokens(this.totalTokens);
            if (this.messages != null) {
                for (MessageData msgData : this.messages) {
                    session.addMessage(msgData.toMessage());
                }
            }
            return session;
        }

        ISessionStore.SessionSummary toSummary() {
            return new ISessionStore.SessionSummary(this.id, this.title, this.projectName, Session.SessionStatus.valueOf(this.status), this.createdAt, this.updatedAt, this.messages != null ? this.messages.size() : 0);
        }
    }
}

