/*
 * Decompiled with CFR 0.152.
 */
package com.example.vibe.core.provider.config;

import com.example.vibe.core.http.HttpClientFactory;
import com.example.vibe.core.internal.VibeCorePlugin;
import com.example.vibe.core.logging.LogSanitizer;
import com.example.vibe.core.logging.VibeLogger;
import com.example.vibe.core.model.LlmMessage;
import com.example.vibe.core.model.LlmRequest;
import com.example.vibe.core.model.LlmResponse;
import com.example.vibe.core.model.LlmStreamChunk;
import com.example.vibe.core.model.ToolCall;
import com.example.vibe.core.model.ToolDefinition;
import com.example.vibe.core.provider.ILlmProvider;
import com.example.vibe.core.provider.LlmProviderException;
import com.example.vibe.core.provider.config.LlmProviderConfig;
import com.google.gson.Gson;
import com.google.gson.JsonArray;
import com.google.gson.JsonElement;
import com.google.gson.JsonObject;
import com.google.gson.JsonParser;
import java.io.IOException;
import java.net.URI;
import java.net.http.HttpClient;
import java.net.http.HttpRequest;
import java.net.http.HttpResponse;
import java.time.Duration;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.concurrent.CancellationException;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.function.Consumer;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import org.eclipse.core.runtime.preferences.IEclipsePreferences;
import org.eclipse.core.runtime.preferences.InstanceScope;

public class DynamicLlmProvider
implements ILlmProvider {
    private static final VibeLogger.CategoryLogger LOG = VibeLogger.forClass(DynamicLlmProvider.class);
    private final LlmProviderConfig config;
    private final HttpClient httpClient;
    private final Gson gson;
    private final AtomicBoolean cancelled = new AtomicBoolean(false);
    private CompletableFuture<?> currentRequest;
    private LlmRequest currentLlmRequest;
    private final Map<Integer, StreamingToolCall> streamingToolCalls = new ConcurrentHashMap<Integer, StreamingToolCall>();

    public DynamicLlmProvider(LlmProviderConfig config) {
        this.config = config;
        this.httpClient = this.createHttpClient();
        this.gson = new Gson();
    }

    @Override
    public String getId() {
        return this.config.getId();
    }

    @Override
    public String getDisplayName() {
        return this.config.getName();
    }

    @Override
    public boolean isConfigured() {
        return this.config.isConfigured();
    }

    @Override
    public boolean supportsStreaming() {
        return this.config.isStreamingEnabled();
    }

    public LlmProviderConfig getConfig() {
        return this.config;
    }

    @Override
    public CompletableFuture<LlmResponse> complete(LlmRequest request) {
        long startTime = System.currentTimeMillis();
        String correlationId = LogSanitizer.newCorrelationId();
        if (!this.isConfigured()) {
            LOG.warn("[%s] Provider not configured: %s", correlationId, this.config.getName());
            return CompletableFuture.failedFuture(new LlmProviderException("Provider not configured: " + this.config.getName()));
        }
        this.cancelled.set(false);
        this.currentLlmRequest = request;
        LOG.info("[%s] DynamicProvider complete: provider=%s, model=%s, messages=%d", correlationId, this.config.getName(), this.config.getModel(), request.getMessages().size());
        String requestBody = this.buildRequestBody(request, false);
        LOG.debug("[%s] Request body: %s", correlationId, requestBody.length() < 5000 ? requestBody : "(truncated, length=" + requestBody.length() + ")");
        HttpRequest httpRequest = this.buildHttpRequest(requestBody);
        this.currentRequest = ((CompletableFuture)this.httpClient.sendAsync(httpRequest, HttpResponse.BodyHandlers.ofString()).thenApply(response -> {
            if (this.cancelled.get()) {
                throw new CancellationException("Request cancelled");
            }
            long duration = System.currentTimeMillis() - startTime;
            LOG.debug("[%s] Response status: %d in %s", correlationId, response.statusCode(), LogSanitizer.formatDuration(duration));
            LOG.debug("[%s] Response body: %s", correlationId, ((String)response.body()).length() < 5000 ? response.body() : "(truncated, length=" + ((String)response.body()).length() + ")");
            return this.parseResponse((HttpResponse<String>)response);
        })).whenComplete((result, error) -> {
            long duration = System.currentTimeMillis() - startTime;
            if (error != null) {
                LOG.error("[%s] DynamicProvider request failed after %s: %s", correlationId, LogSanitizer.formatDuration(duration), error.getMessage());
            } else {
                LOG.info("[%s] DynamicProvider response received in %s", correlationId, LogSanitizer.formatDuration(duration));
            }
        });
        return this.currentRequest.thenApply(obj -> (LlmResponse)obj);
    }

    @Override
    public void streamComplete(LlmRequest request, Consumer<LlmStreamChunk> consumer) {
        block10: {
            long startTime = System.currentTimeMillis();
            String correlationId = LogSanitizer.newCorrelationId();
            if (!this.isConfigured()) {
                LOG.warn("[%s] Provider not configured: %s (stream)", correlationId, this.config.getName());
                throw new LlmProviderException("Provider not configured: " + this.config.getName());
            }
            this.cancelled.set(false);
            this.streamingToolCalls.clear();
            LOG.info("[%s] DynamicProvider streamComplete: provider=%s, model=%s, messages=%d", correlationId, this.config.getName(), this.config.getModel(), request.getMessages().size());
            String requestBody = this.buildRequestBody(request, true);
            HttpRequest httpRequest = this.buildHttpRequest(requestBody);
            try {
                HttpResponse<Stream<String>> response = this.httpClient.send(httpRequest, HttpResponse.BodyHandlers.ofLines());
                if (this.cancelled.get()) {
                    LOG.debug("[%s] Stream cancelled before processing", correlationId);
                    return;
                }
                if (response.statusCode() != 200) {
                    String errorBody = response.body().collect(Collectors.joining("\n"));
                    String errorType = this.extractErrorType(errorBody);
                    LOG.error("[%s] Stream API error: status=%d, errorType=%s", correlationId, response.statusCode(), errorType);
                    throw new LlmProviderException("API error: " + response.statusCode() + " - " + errorBody, null, response.statusCode(), errorType);
                }
                String[] streamFinishReason = new String[]{"stop"};
                response.body().forEach(line -> {
                    if (this.cancelled.get()) {
                        return;
                    }
                    String finishReason = this.processStreamLine((String)line, consumer);
                    if (finishReason != null) {
                        stringArray[0] = finishReason;
                    }
                });
                if (!this.streamingToolCalls.isEmpty() && !this.cancelled.get()) {
                    ArrayList<ToolCall> toolCalls = new ArrayList<ToolCall>();
                    this.streamingToolCalls.entrySet().stream().sorted(Map.Entry.comparingByKey()).forEach(entry -> {
                        ToolCall tc = ((StreamingToolCall)entry.getValue()).toToolCall();
                        if (tc.getName() == null || tc.getName().trim().isEmpty()) {
                            LOG.warn("[%s] Skipping invalid tool call with empty name at index %d, args: %s", correlationId, entry.getKey(), tc.getArguments());
                            return;
                        }
                        toolCalls.add(tc);
                    });
                    if (!toolCalls.isEmpty()) {
                        LOG.debug("[%s] Stream completed with %d valid tool calls", correlationId, toolCalls.size());
                        for (ToolCall tc : toolCalls) {
                            LOG.debug("[%s]   Tool call: %s(%s)", correlationId, tc.getName(), tc.getArguments().length() > 100 ? tc.getArguments().substring(0, 100) + "..." : tc.getArguments());
                        }
                        consumer.accept(LlmStreamChunk.toolCalls(toolCalls));
                    } else {
                        LOG.warn("[%s] All %d streamed tool calls were invalid, ignoring", correlationId, this.streamingToolCalls.size());
                    }
                }
                if (!this.cancelled.get()) {
                    consumer.accept(LlmStreamChunk.complete(streamFinishReason[0]));
                }
                long duration = System.currentTimeMillis() - startTime;
                LOG.info("[%s] DynamicProvider stream completed in %s", correlationId, LogSanitizer.formatDuration(duration));
            }
            catch (IOException | InterruptedException e) {
                long duration = System.currentTimeMillis() - startTime;
                if (this.cancelled.get()) break block10;
                LOG.error("[%s] DynamicProvider stream failed after %s: %s", correlationId, LogSanitizer.formatDuration(duration), e.getMessage());
                throw new LlmProviderException("Stream request failed: " + e.getMessage(), e);
            }
        }
    }

    private String processStreamLine(String line, Consumer<LlmStreamChunk> consumer) {
        if (line == null || line.isEmpty()) {
            return null;
        }
        if (line.startsWith("data: ")) {
            String data = line.substring(6).trim();
            if ("[DONE]".equals(data)) {
                return null;
            }
            try {
                String reasoning;
                JsonObject json = JsonParser.parseString((String)data).getAsJsonObject();
                String chunk = this.extractChunkContent(json);
                if (chunk != null && !chunk.isEmpty()) {
                    consumer.accept(LlmStreamChunk.content(chunk));
                }
                if ((reasoning = this.extractReasoningContent(json)) != null && !reasoning.isEmpty()) {
                    consumer.accept(LlmStreamChunk.reasoning(reasoning));
                }
                this.extractAndAccumulateToolCalls(json);
                return this.extractFinishReason(json);
            }
            catch (Exception e) {
                VibeCorePlugin.logWarn("Failed to parse stream chunk: " + e.getMessage());
            }
        }
        return null;
    }

    private void extractAndAccumulateToolCalls(JsonObject json) {
        JsonArray choices = json.getAsJsonArray("choices");
        if (choices == null || choices.size() == 0) {
            return;
        }
        JsonObject choice = choices.get(0).getAsJsonObject();
        JsonObject delta = choice.getAsJsonObject("delta");
        if (delta == null || !delta.has("tool_calls")) {
            return;
        }
        JsonArray toolCallsArray = delta.getAsJsonArray("tool_calls");
        for (JsonElement element : toolCallsArray) {
            JsonObject tcObj = element.getAsJsonObject();
            int index = tcObj.has("index") ? tcObj.get("index").getAsInt() : 0;
            StreamingToolCall accumulator = this.streamingToolCalls.computeIfAbsent(index, k -> new StreamingToolCall());
            if (tcObj.has("id") && !tcObj.get("id").isJsonNull()) {
                accumulator.id = String.valueOf(accumulator.id) + tcObj.get("id").getAsString();
            }
            if (!tcObj.has("function")) continue;
            JsonObject function = tcObj.getAsJsonObject("function");
            if (function.has("name") && !function.get("name").isJsonNull()) {
                accumulator.name = String.valueOf(accumulator.name) + function.get("name").getAsString();
            }
            if (!function.has("arguments") || function.get("arguments").isJsonNull()) continue;
            accumulator.arguments.append(function.get("arguments").getAsString());
        }
    }

    private String extractFinishReason(JsonObject json) {
        JsonArray choices = json.getAsJsonArray("choices");
        if (choices == null || choices.size() == 0) {
            return null;
        }
        JsonObject choice = choices.get(0).getAsJsonObject();
        if (choice.has("finish_reason") && !choice.get("finish_reason").isJsonNull()) {
            return choice.get("finish_reason").getAsString();
        }
        return null;
    }

    @Override
    public void cancel() {
        this.cancelled.set(true);
        if (this.currentRequest != null) {
            this.currentRequest.cancel(true);
        }
    }

    @Override
    public void dispose() {
        this.cancel();
    }

    private HttpRequest buildHttpRequest(String body) {
        String url = this.config.getChatEndpointUrl();
        LOG.info("=== HTTP REQUEST BUILD ===");
        LOG.info("URL: %s", url);
        LOG.info("Base URL from config: %s", this.config.getBaseUrl());
        LOG.info("Provider type: %s", new Object[]{this.config.getType()});
        HttpRequest.Builder builder = HttpRequest.newBuilder().uri(URI.create(url)).timeout(Duration.ofSeconds(this.getRequestTimeoutSeconds())).header("Content-Type", "application/json");
        String apiKey = this.config.getApiKey();
        if (apiKey != null && !apiKey.isEmpty()) {
            LOG.info("API Key length: %d, first 8 chars: %s...", apiKey.length(), apiKey.substring(0, Math.min(8, apiKey.length())));
            switch (this.config.getType()) {
                case ANTHROPIC: {
                    builder.header("x-api-key", apiKey);
                    builder.header("anthropic-version", "2023-06-01");
                    LOG.info("Auth: x-api-key header (Anthropic)");
                    break;
                }
                default: {
                    builder.header("Authorization", "Bearer " + apiKey);
                    LOG.info("Auth: Bearer token (OpenAI compatible)");
                    break;
                }
            }
        } else {
            LOG.warn("No API key configured!");
        }
        this.config.getCustomHeaders().forEach((key, value) -> {
            LOG.info("Custom header: %s = %s", key, value);
            builder.header((String)key, (String)value);
        });
        builder.POST(HttpRequest.BodyPublishers.ofString(body));
        LOG.info("=== END HTTP REQUEST BUILD ===");
        return builder.build();
    }

    private HttpClient createHttpClient() {
        HttpClientFactory factory;
        VibeCorePlugin plugin = VibeCorePlugin.getDefault();
        if (plugin != null && (factory = plugin.getHttpClientFactory()) != null) {
            return factory.getSharedClient();
        }
        return HttpClient.newBuilder().version(HttpClient.Version.HTTP_2).connectTimeout(Duration.ofSeconds(60L)).build();
    }

    private int getRequestTimeoutSeconds() {
        IEclipsePreferences prefs = InstanceScope.INSTANCE.getNode("com.example.vibe.core");
        return prefs.getInt("requestTimeout", 180);
    }

    private String buildRequestBody(LlmRequest request, boolean stream) {
        new JsonObject();
        switch (this.config.getType()) {
            case ANTHROPIC: {
                return this.buildAnthropicRequestBody(request, stream);
            }
            case OLLAMA: {
                return this.buildOllamaRequestBody(request, stream);
            }
        }
        return this.buildOpenAiRequestBody(request, stream);
    }

    private String buildOpenAiRequestBody(LlmRequest request, boolean stream) {
        String model;
        JsonObject body = new JsonObject();
        String requestModel = request.getModel();
        String string = model = requestModel != null && !requestModel.isEmpty() ? requestModel : this.config.getModel();
        if (model == null || model.isEmpty()) {
            model = "auto";
        }
        body.addProperty("model", model);
        body.addProperty("max_tokens", (Number)this.config.getMaxTokens());
        body.addProperty("stream", Boolean.valueOf(stream));
        JsonArray messages = new JsonArray();
        for (LlmMessage msg : request.getMessages()) {
            messages.add((JsonElement)this.serializeMessage(msg));
        }
        body.add("messages", (JsonElement)messages);
        if (request.hasTools()) {
            JsonArray tools = new JsonArray();
            for (ToolDefinition tool : request.getTools()) {
                tools.add((JsonElement)this.serializeToolDefinition(tool));
            }
            body.add("tools", (JsonElement)tools);
            LOG.debug("Added %d tools to request", request.getTools().size());
            if (request.getToolChoice() != null) {
                String toolChoice = this.serializeToolChoice(request.getToolChoice());
                body.addProperty("tool_choice", toolChoice);
            }
        }
        return this.gson.toJson((JsonElement)body);
    }

    private JsonObject serializeMessage(LlmMessage msg) {
        JsonObject msgObj = new JsonObject();
        msgObj.addProperty("role", msg.getRole().getValue());
        if (msg.getRole() == LlmMessage.Role.TOOL) {
            msgObj.addProperty("tool_call_id", msg.getToolCallId());
            msgObj.addProperty("content", msg.getContent());
        } else if (msg.hasToolCalls()) {
            if (msg.getContent() != null && !msg.getContent().isEmpty()) {
                msgObj.addProperty("content", msg.getContent());
            } else {
                msgObj.add("content", null);
            }
            JsonArray toolCalls = new JsonArray();
            for (ToolCall call : msg.getToolCalls()) {
                JsonObject callObj = new JsonObject();
                callObj.addProperty("id", call.getId());
                callObj.addProperty("type", "function");
                JsonObject functionObj = new JsonObject();
                functionObj.addProperty("name", call.getName());
                functionObj.addProperty("arguments", call.getArguments());
                callObj.add("function", (JsonElement)functionObj);
                toolCalls.add((JsonElement)callObj);
            }
            msgObj.add("tool_calls", (JsonElement)toolCalls);
        } else {
            msgObj.addProperty("content", msg.getContent());
        }
        return msgObj;
    }

    private JsonObject serializeToolDefinition(ToolDefinition tool) {
        JsonObject toolObj = new JsonObject();
        toolObj.addProperty("type", "function");
        JsonObject functionObj = new JsonObject();
        functionObj.addProperty("name", tool.getName());
        functionObj.addProperty("description", tool.getDescription());
        try {
            JsonElement params = JsonParser.parseString((String)tool.getParametersSchema());
            functionObj.add("parameters", params);
        }
        catch (Exception exception) {
            functionObj.add("parameters", (JsonElement)new JsonObject());
        }
        toolObj.add("function", (JsonElement)functionObj);
        return toolObj;
    }

    private String serializeToolChoice(LlmRequest.ToolChoice choice) {
        switch (choice) {
            case AUTO: {
                return "auto";
            }
            case REQUIRED: {
                return "required";
            }
            case NONE: {
                return "none";
            }
        }
        return "auto";
    }

    private String buildAnthropicRequestBody(LlmRequest request, boolean stream) {
        JsonObject body = new JsonObject();
        body.addProperty("model", this.config.getModel());
        body.addProperty("max_tokens", (Number)this.config.getMaxTokens());
        body.addProperty("stream", Boolean.valueOf(stream));
        JsonArray messages = new JsonArray();
        for (LlmMessage msg : request.getMessages()) {
            if (msg.getRole() == LlmMessage.Role.SYSTEM) {
                body.addProperty("system", msg.getContent());
                continue;
            }
            JsonObject msgObj = new JsonObject();
            msgObj.addProperty("role", msg.getRole().getValue());
            msgObj.addProperty("content", msg.getContent());
            messages.add((JsonElement)msgObj);
        }
        body.add("messages", (JsonElement)messages);
        return this.gson.toJson((JsonElement)body);
    }

    private String buildOllamaRequestBody(LlmRequest request, boolean stream) {
        JsonObject body = new JsonObject();
        body.addProperty("model", this.config.getModel());
        body.addProperty("stream", Boolean.valueOf(stream));
        JsonArray messages = new JsonArray();
        for (LlmMessage msg : request.getMessages()) {
            JsonObject msgObj = new JsonObject();
            msgObj.addProperty("role", msg.getRole().getValue());
            msgObj.addProperty("content", msg.getContent());
            messages.add((JsonElement)msgObj);
        }
        body.add("messages", (JsonElement)messages);
        return this.gson.toJson((JsonElement)body);
    }

    private LlmResponse parseResponse(HttpResponse<String> response) {
        if (response.statusCode() != 200) {
            String body = response.body();
            String errorType = this.extractErrorType(body);
            throw new LlmProviderException("API error: " + response.statusCode() + " - " + body, null, response.statusCode(), errorType);
        }
        try {
            JsonObject json = JsonParser.parseString((String)response.body()).getAsJsonObject();
            switch (this.config.getType()) {
                case ANTHROPIC: {
                    return this.parseAnthropicResponse(json);
                }
                case OLLAMA: {
                    return this.parseOllamaResponse(json);
                }
            }
            return this.parseOpenAiResponse(json);
        }
        catch (Exception e) {
            throw new LlmProviderException("Failed to parse response: " + e.getMessage(), e);
        }
    }

    private LlmResponse parseOpenAiResponse(JsonObject json) {
        JsonArray details;
        JsonArray choices = json.getAsJsonArray("choices");
        if (choices == null || choices.size() == 0) {
            throw new LlmProviderException("No choices in response");
        }
        JsonObject choice = choices.get(0).getAsJsonObject();
        JsonObject message = choice.getAsJsonObject("message");
        LOG.debug("Message keys: %s", message.keySet());
        String content = null;
        if (message.has("content") && !message.get("content").isJsonNull()) {
            content = message.get("content").getAsString();
        }
        String reasoningContent = null;
        if (message.has("reasoning_content") && !message.get("reasoning_content").isJsonNull()) {
            reasoningContent = message.get("reasoning_content").getAsString();
        }
        if (reasoningContent == null && message.has("reasoning_details") && (details = message.getAsJsonArray("reasoning_details")) != null) {
            StringBuilder rb = new StringBuilder();
            for (JsonElement el : details) {
                JsonObject detail;
                if (!el.isJsonObject() || !(detail = el.getAsJsonObject()).has("text") || detail.get("text").isJsonNull()) continue;
                rb.append(detail.get("text").getAsString());
            }
            if (rb.length() > 0) {
                reasoningContent = rb.toString();
            }
        }
        if (reasoningContent != null) {
            LOG.debug("Parsed reasoning_content: %d chars", reasoningContent.length());
        }
        String finishReason = choice.has("finish_reason") && !choice.get("finish_reason").isJsonNull() ? choice.get("finish_reason").getAsString() : "stop";
        List<ToolCall> toolCalls = null;
        if (message.has("tool_calls")) {
            LOG.debug("tool_calls found in message");
            JsonArray toolCallsJson = message.getAsJsonArray("tool_calls");
            toolCalls = this.parseToolCalls(toolCallsJson);
            LOG.debug("Parsed %d tool calls", toolCalls.size());
            for (ToolCall tc : toolCalls) {
                LOG.debug("  Tool call: %s(%s)", tc.getName(), tc.getArguments());
            }
            if (!toolCalls.isEmpty() && !"tool_calls".equals(finishReason)) {
                finishReason = "tool_use";
            }
        } else {
            LOG.debug("tool_calls NOT found in message");
        }
        LOG.debug("Response parsed: finishReason=%s, hasContent=%b, toolCalls=%d", finishReason, content != null && !content.isEmpty(), toolCalls != null ? toolCalls.size() : 0);
        return new LlmResponse(content, this.config.getModel(), null, finishReason, toolCalls, reasoningContent);
    }

    private List<ToolCall> parseToolCalls(JsonArray toolCallsJson) {
        ArrayList<ToolCall> toolCalls = new ArrayList<ToolCall>();
        for (JsonElement element : toolCallsJson) {
            JsonObject callObj = element.getAsJsonObject();
            String id = callObj.get("id").getAsString();
            JsonObject function = callObj.getAsJsonObject("function");
            String name = function.get("name").getAsString();
            String arguments = function.get("arguments").getAsString();
            toolCalls.add(new ToolCall(id, name, arguments));
        }
        return toolCalls;
    }

    private LlmResponse parseAnthropicResponse(JsonObject json) {
        JsonArray content = json.getAsJsonArray("content");
        if (content == null || content.size() == 0) {
            throw new LlmProviderException("No content in response");
        }
        StringBuilder sb = new StringBuilder();
        int i = 0;
        while (i < content.size()) {
            JsonObject block = content.get(i).getAsJsonObject();
            if ("text".equals(block.get("type").getAsString())) {
                sb.append(block.get("text").getAsString());
            }
            ++i;
        }
        String stopReason = json.has("stop_reason") ? json.get("stop_reason").getAsString() : "end_turn";
        return new LlmResponse(sb.toString(), this.config.getModel(), null, stopReason);
    }

    private LlmResponse parseOllamaResponse(JsonObject json) {
        JsonObject message = json.getAsJsonObject("message");
        String content = message.get("content").getAsString();
        boolean done = json.has("done") && json.get("done").getAsBoolean();
        return new LlmResponse(content, this.config.getModel(), null, done ? "stop" : "length");
    }

    private String extractErrorType(String body) {
        if (body == null || body.isEmpty()) {
            return null;
        }
        try {
            JsonObject json = JsonParser.parseString((String)body).getAsJsonObject();
            if (json.has("error")) {
                JsonElement errorElement = json.get("error");
                if (errorElement.isJsonObject()) {
                    JsonObject errorObj = errorElement.getAsJsonObject();
                    if (errorObj.has("type") && !errorObj.get("type").isJsonNull()) {
                        return errorObj.get("type").getAsString();
                    }
                    if (errorObj.has("code") && !errorObj.get("code").isJsonNull()) {
                        return errorObj.get("code").getAsString();
                    }
                } else if (errorElement.isJsonPrimitive()) {
                    return errorElement.getAsString();
                }
            }
        }
        catch (Exception e) {
            LOG.debug("Failed to extract error type from response body: %s", e.getMessage());
        }
        return null;
    }

    private String extractChunkContent(JsonObject json) {
        switch (this.config.getType()) {
            case ANTHROPIC: {
                JsonObject delta;
                if (!json.has("delta") || !(delta = json.getAsJsonObject("delta")).has("text")) break;
                return delta.get("text").getAsString();
            }
            case OLLAMA: {
                JsonObject message;
                if (!json.has("message") || !(message = json.getAsJsonObject("message")).has("content")) break;
                return message.get("content").getAsString();
            }
            default: {
                JsonObject delta;
                JsonObject choice;
                JsonArray choices;
                if (!json.has("choices") || (choices = json.getAsJsonArray("choices")).size() <= 0 || !(choice = choices.get(0).getAsJsonObject()).has("delta") || !(delta = choice.getAsJsonObject("delta")).has("content")) break;
                return delta.get("content").getAsString();
            }
        }
        return null;
    }

    private String extractReasoningContent(JsonObject json) {
        JsonArray details;
        JsonArray choices = json.getAsJsonArray("choices");
        if (choices == null || choices.size() == 0) {
            return null;
        }
        JsonObject choice = choices.get(0).getAsJsonObject();
        JsonObject delta = choice.getAsJsonObject("delta");
        if (delta == null) {
            return null;
        }
        if (delta.has("reasoning_content") && !delta.get("reasoning_content").isJsonNull()) {
            return delta.get("reasoning_content").getAsString();
        }
        if (delta.has("reasoning_details") && (details = delta.getAsJsonArray("reasoning_details")) != null) {
            StringBuilder rb = new StringBuilder();
            for (JsonElement el : details) {
                JsonObject detail;
                if (!el.isJsonObject() || !(detail = el.getAsJsonObject()).has("text") || detail.get("text").isJsonNull()) continue;
                rb.append(detail.get("text").getAsString());
            }
            if (rb.length() > 0) {
                return rb.toString();
            }
        }
        return null;
    }

    private static class StreamingToolCall {
        String id = "";
        String name = "";
        StringBuilder arguments = new StringBuilder();

        private StreamingToolCall() {
        }

        ToolCall toToolCall() {
            return new ToolCall(this.id, this.name, this.arguments.toString());
        }
    }
}

