/*
 * Decompiled with CFR 0.152.
 */
package com.example.vibe.core.mcp.transport;

import com.example.vibe.core.logging.VibeLogger;
import com.example.vibe.core.mcp.model.McpError;
import com.example.vibe.core.mcp.model.McpException;
import com.example.vibe.core.mcp.model.McpMessage;
import com.example.vibe.core.mcp.transport.IMcpTransport;
import com.google.gson.Gson;
import com.google.gson.GsonBuilder;
import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.Closeable;
import java.io.File;
import java.io.IOException;
import java.io.InputStreamReader;
import java.io.OutputStreamWriter;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicLong;
import java.util.function.Consumer;

public class McpStdioTransport
implements IMcpTransport {
    private static final VibeLogger.CategoryLogger LOG = VibeLogger.forClass(McpStdioTransport.class);
    private final String command;
    private final List<String> args;
    private final Map<String, String> env;
    private final File workingDirectory;
    private final int requestTimeoutMs;
    private Process process;
    private BufferedReader stdout;
    private BufferedWriter stdin;
    private BufferedReader stderr;
    private Thread readerThread;
    private Thread stderrThread;
    private volatile boolean running = false;
    private final Map<String, CompletableFuture<McpMessage>> pendingRequests = new ConcurrentHashMap<String, CompletableFuture<McpMessage>>();
    private final AtomicLong requestIdCounter = new AtomicLong(1L);
    private Consumer<McpMessage> notificationHandler;
    private final Gson gson = new GsonBuilder().create();

    public McpStdioTransport(String command, List<String> args, Map<String, String> env, File workingDirectory, int requestTimeoutMs) {
        this.command = command;
        this.args = args != null ? new ArrayList<String>(args) : new ArrayList();
        this.env = env != null ? new HashMap<String, String>(env) : new HashMap();
        this.workingDirectory = workingDirectory;
        this.requestTimeoutMs = requestTimeoutMs > 0 ? requestTimeoutMs : 60000;
    }

    @Override
    public void connect() throws IOException {
        Map<String, String> resolvedEnv = this.resolveEnvironment(this.env);
        String resolvedCommand = this.resolveCommandPath(this.command, resolvedEnv);
        ArrayList<String> fullCommand = new ArrayList<String>();
        fullCommand.add(resolvedCommand);
        fullCommand.addAll(this.args);
        ProcessBuilder pb = new ProcessBuilder(fullCommand);
        pb.environment().putAll(resolvedEnv);
        if (this.workingDirectory != null) {
            pb.directory(this.workingDirectory);
        }
        pb.redirectErrorStream(false);
        LOG.info("Starting MCP process: %s", String.join((CharSequence)" ", fullCommand));
        this.process = pb.start();
        this.running = true;
        this.stdout = new BufferedReader(new InputStreamReader(this.process.getInputStream(), StandardCharsets.UTF_8));
        this.stdin = new BufferedWriter(new OutputStreamWriter(this.process.getOutputStream(), StandardCharsets.UTF_8));
        this.stderr = new BufferedReader(new InputStreamReader(this.process.getErrorStream(), StandardCharsets.UTF_8));
        this.readerThread = new Thread(this::readLoop, "MCP-STDIO-Reader-" + this.command);
        this.readerThread.setDaemon(true);
        this.readerThread.start();
        this.stderrThread = new Thread(this::logStderr, "MCP-STDIO-Stderr-" + this.command);
        this.stderrThread.setDaemon(true);
        this.stderrThread.start();
        LOG.info("MCP STDIO transport connected: %s", this.command);
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    private void readLoop() {
        try {
            try {
                while (this.running) {
                    String line = this.stdout.readLine();
                    if (line == null) {
                        return;
                    }
                    try {
                        McpMessage message = (McpMessage)this.gson.fromJson(line, McpMessage.class);
                        this.handleMessage(message);
                    }
                    catch (Exception e) {
                        LOG.warn("Failed to parse MCP message: %s - line: %s", e.getMessage(), line);
                    }
                }
                return;
            }
            catch (IOException e) {
                if (this.running) {
                    LOG.error("MCP STDIO read error", e);
                }
                this.running = false;
                return;
            }
        }
        finally {
            this.running = false;
        }
    }

    private void logStderr() {
        try {
            String line;
            while (this.running && (line = this.stderr.readLine()) != null) {
                LOG.debug("[MCP stderr] %s", line);
            }
        }
        catch (IOException iOException) {}
    }

    private void handleMessage(McpMessage message) {
        if (message.isResponse()) {
            String id = message.getId();
            CompletableFuture<McpMessage> future = this.pendingRequests.remove(id);
            if (future != null) {
                if (message.getError() != null) {
                    future.completeExceptionally(new McpException(message.getError()));
                } else {
                    future.complete(message);
                }
            } else {
                LOG.warn("Received response for unknown request ID: %s", id);
            }
        } else if (message.isNotification()) {
            if (this.notificationHandler != null) {
                this.notificationHandler.accept(message);
            }
        } else if (message.isRequest()) {
            LOG.warn("Received unsupported server request: %s", message.getMethod());
            this.sendErrorResponse(message.getId(), -32601, "Method not found: " + message.getMethod());
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void sendErrorResponse(String id, int code, String errorMessage) {
        try {
            McpMessage response = new McpMessage();
            response.setId(id);
            response.setError(new McpError(code, errorMessage, null));
            String json = this.gson.toJson((Object)response);
            BufferedWriter bufferedWriter = this.stdin;
            synchronized (bufferedWriter) {
                this.stdin.write(json);
                this.stdin.newLine();
                this.stdin.flush();
            }
        }
        catch (IOException e) {
            LOG.warn("Failed to send error response", e);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public CompletableFuture<McpMessage> send(McpMessage message) {
        if (!this.isConnected()) {
            return CompletableFuture.failedFuture(new IOException("Transport not connected"));
        }
        String id = String.valueOf(this.requestIdCounter.getAndIncrement());
        message.setId(id);
        CompletableFuture<McpMessage> future = new CompletableFuture<McpMessage>();
        this.pendingRequests.put(id, future);
        try {
            String json = this.gson.toJson((Object)message);
            BufferedWriter bufferedWriter = this.stdin;
            synchronized (bufferedWriter) {
                this.stdin.write(json);
                this.stdin.newLine();
                this.stdin.flush();
            }
            LOG.debug("Sent MCP request: %s", message);
        }
        catch (IOException e) {
            this.pendingRequests.remove(id);
            future.completeExceptionally(e);
            return future;
        }
        return future.orTimeout(this.requestTimeoutMs, TimeUnit.MILLISECONDS).whenComplete((result, error) -> this.pendingRequests.remove(id));
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public CompletableFuture<Void> sendNotification(McpMessage message) {
        if (!this.isConnected()) {
            return CompletableFuture.failedFuture(new IOException("Transport not connected"));
        }
        try {
            String json = this.gson.toJson((Object)message);
            BufferedWriter bufferedWriter = this.stdin;
            synchronized (bufferedWriter) {
                this.stdin.write(json);
                this.stdin.newLine();
                this.stdin.flush();
            }
            LOG.debug("Sent MCP notification: %s", message);
            return CompletableFuture.completedFuture(null);
        }
        catch (IOException e) {
            return CompletableFuture.failedFuture(e);
        }
    }

    @Override
    public void disconnect() {
        this.running = false;
        this.pendingRequests.forEach((id, future) -> {
            boolean bl = future.completeExceptionally(new IOException("Transport disconnected"));
        });
        this.pendingRequests.clear();
        this.closeQuietly(this.stdin);
        this.closeQuietly(this.stdout);
        this.closeQuietly(this.stderr);
        if (this.process != null && this.process.isAlive()) {
            this.process.destroy();
            try {
                if (!this.process.waitFor(5L, TimeUnit.SECONDS)) {
                    this.process.destroyForcibly();
                }
            }
            catch (InterruptedException interruptedException) {
                this.process.destroyForcibly();
                Thread.currentThread().interrupt();
            }
        }
        if (this.readerThread != null) {
            this.readerThread.interrupt();
            try {
                this.readerThread.join(1000L);
            }
            catch (InterruptedException interruptedException) {
                Thread.currentThread().interrupt();
            }
        }
        if (this.stderrThread != null) {
            this.stderrThread.interrupt();
            try {
                this.stderrThread.join(1000L);
            }
            catch (InterruptedException interruptedException) {
                Thread.currentThread().interrupt();
            }
        }
        LOG.info("MCP STDIO transport disconnected: %s", this.command);
    }

    private void closeQuietly(Closeable closeable) {
        if (closeable != null) {
            try {
                closeable.close();
            }
            catch (IOException iOException) {}
        }
    }

    @Override
    public void close() {
        this.disconnect();
    }

    @Override
    public boolean isConnected() {
        return this.running && this.process != null && this.process.isAlive();
    }

    @Override
    public void setNotificationHandler(Consumer<McpMessage> handler) {
        this.notificationHandler = handler;
    }

    private String resolveCommandPath(String cmd, Map<String, String> resolvedEnv) {
        String[] pathDirs;
        File cmdFile = new File(cmd);
        if (cmdFile.isAbsolute() && cmdFile.canExecute()) {
            return cmd;
        }
        String pathKey = this.isWindows() ? "Path" : "PATH";
        String pathValue = resolvedEnv.get(pathKey);
        if (pathValue == null) {
            pathValue = resolvedEnv.get("PATH");
        }
        if (pathValue == null) {
            LOG.warn("No PATH found in environment, using command as-is: %s", cmd);
            return cmd;
        }
        String[] stringArray = pathDirs = pathValue.split(File.pathSeparator);
        int n = pathDirs.length;
        int n2 = 0;
        while (n2 < n) {
            String dir = stringArray[n2];
            if (!dir.isEmpty()) {
                if (this.isWindows()) {
                    String[] extensions;
                    String[] stringArray2 = extensions = new String[]{".exe", ".cmd", ".bat", ".com", ""};
                    int n3 = extensions.length;
                    int n4 = 0;
                    while (n4 < n3) {
                        String ext = stringArray2[n4];
                        candidate = new File(dir, cmd + ext);
                        if (candidate.isFile() && candidate.canExecute()) {
                            LOG.debug("Resolved command '%s' to '%s'", cmd, candidate.getAbsolutePath());
                            return candidate.getAbsolutePath();
                        }
                        ++n4;
                    }
                } else {
                    candidate = new File(dir, cmd);
                    if (candidate.isFile() && candidate.canExecute()) {
                        LOG.debug("Resolved command '%s' to '%s'", cmd, candidate.getAbsolutePath());
                        return candidate.getAbsolutePath();
                    }
                }
            }
            ++n2;
        }
        LOG.warn("Could not resolve command '%s' in PATH, using as-is", cmd);
        return cmd;
    }

    private Map<String, String> resolveEnvironment(Map<String, String> env) {
        String pathKey;
        HashMap<String, String> resolved = new HashMap<String, String>();
        for (Map.Entry<String, String> entry : env.entrySet()) {
            String envVar;
            String value = entry.getValue();
            if (value != null && value.startsWith("${") && value.endsWith("}") && (value = System.getenv(envVar = value.substring(2, value.length() - 1))) == null) {
                LOG.warn("Environment variable not found: %s", envVar);
                value = "";
            }
            resolved.put(entry.getKey(), value);
        }
        String string = pathKey = this.isWindows() ? "Path" : "PATH";
        if (!resolved.containsKey(pathKey) && !resolved.containsKey("PATH")) {
            String currentPath = System.getenv(pathKey);
            if (currentPath == null) {
                currentPath = System.getenv("PATH");
            }
            if (currentPath == null) {
                currentPath = "";
            }
            List<String> additionalPaths = this.getAdditionalPaths();
            StringBuilder pathBuilder = new StringBuilder();
            for (String path : additionalPaths) {
                File dir = new File(path);
                if (!dir.isDirectory()) continue;
                if (pathBuilder.length() > 0) {
                    pathBuilder.append(File.pathSeparator);
                }
                pathBuilder.append(path);
            }
            if (!currentPath.isEmpty()) {
                if (pathBuilder.length() > 0) {
                    pathBuilder.append(File.pathSeparator);
                }
                pathBuilder.append(currentPath);
            }
            resolved.put(pathKey, pathBuilder.toString());
            LOG.debug("Resolved %s for MCP process: %s", pathKey, pathBuilder.toString());
        }
        return resolved;
    }

    private boolean isWindows() {
        return System.getProperty("os.name", "").toLowerCase().contains("win");
    }

    private boolean isMac() {
        return System.getProperty("os.name", "").toLowerCase().contains("mac");
    }

    private List<String> getAdditionalPaths() {
        ArrayList<String> paths = new ArrayList<String>();
        String userHome = System.getProperty("user.home");
        if (this.isWindows()) {
            File[] nodeDirs;
            File nvmDir;
            String appData = System.getenv("APPDATA");
            String localAppData = System.getenv("LOCALAPPDATA");
            String programFiles = System.getenv("ProgramFiles");
            String programFilesX86 = System.getenv("ProgramFiles(x86)");
            if (appData != null) {
                paths.add(appData + "\\npm");
                paths.add(appData + "\\nvm");
            }
            if (programFiles != null) {
                paths.add(programFiles + "\\nodejs");
            }
            if (programFilesX86 != null) {
                paths.add(programFilesX86 + "\\nodejs");
            }
            if (appData != null && (nvmDir = new File(appData + "\\nvm")).isDirectory() && (nodeDirs = nvmDir.listFiles(f -> f.isDirectory() && f.getName().startsWith("v"))) != null) {
                File[] fileArray = nodeDirs;
                int n = nodeDirs.length;
                int n2 = 0;
                while (n2 < n) {
                    File nodeDir = fileArray[n2];
                    paths.add(nodeDir.getAbsolutePath());
                    ++n2;
                }
            }
            if (localAppData != null) {
                paths.add(localAppData + "\\Programs\\Python\\Python311");
                paths.add(localAppData + "\\Programs\\Python\\Python310");
                paths.add(localAppData + "\\Programs\\Python\\Python39");
            }
            if (userHome != null) {
                paths.add(userHome + "\\.local\\bin");
            }
        } else {
            if (this.isMac()) {
                paths.add("/opt/homebrew/bin");
                paths.add("/opt/homebrew/sbin");
                paths.add("/usr/local/bin");
                paths.add("/usr/local/sbin");
            }
            paths.add("/usr/local/bin");
            paths.add("/usr/bin");
            paths.add("/bin");
            paths.add("/snap/bin");
            if (userHome != null) {
                File[] nodeDirs;
                File fnmDir;
                int n;
                File[] nodeDirs2;
                paths.add(userHome + "/.local/bin");
                paths.add(userHome + "/bin");
                File nvmDir = new File(userHome + "/.nvm/versions/node");
                if (nvmDir.isDirectory() && (nodeDirs2 = nvmDir.listFiles(f -> f.isDirectory() && f.getName().startsWith("v"))) != null) {
                    File[] fileArray = nodeDirs2;
                    n = nodeDirs2.length;
                    int programFilesX86 = 0;
                    while (programFilesX86 < n) {
                        File nodeDir = fileArray[programFilesX86];
                        paths.add(nodeDir.getAbsolutePath() + "/bin");
                        ++programFilesX86;
                    }
                }
                if ((fnmDir = new File(userHome + "/.fnm/node-versions")).isDirectory() && (nodeDirs = fnmDir.listFiles(File::isDirectory)) != null) {
                    File[] fileArray = nodeDirs;
                    int n3 = nodeDirs.length;
                    n = 0;
                    while (n < n3) {
                        File nodeDir = fileArray[n];
                        paths.add(nodeDir.getAbsolutePath() + "/installation/bin");
                        ++n;
                    }
                }
                paths.add(userHome + "/.volta/bin");
                paths.add(userHome + "/.cargo/bin");
                paths.add(userHome + "/go/bin");
            }
        }
        return paths;
    }
}

