/*
 * Decompiled with CFR 0.152.
 */
package com.example.vibe.core.mcp.client;

import com.example.vibe.core.logging.VibeLogger;
import com.example.vibe.core.mcp.model.McpMessage;
import com.example.vibe.core.mcp.model.McpPrompt;
import com.example.vibe.core.mcp.model.McpPromptResult;
import com.example.vibe.core.mcp.model.McpResource;
import com.example.vibe.core.mcp.model.McpResourceContent;
import com.example.vibe.core.mcp.model.McpServerCapabilities;
import com.example.vibe.core.mcp.model.McpTool;
import com.example.vibe.core.mcp.model.McpToolResult;
import com.example.vibe.core.mcp.transport.IMcpTransport;
import com.google.gson.Gson;
import com.google.gson.JsonArray;
import com.google.gson.JsonElement;
import com.google.gson.JsonObject;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.CompletionStage;
import java.util.function.Consumer;

public class McpClient
implements AutoCloseable {
    private static final VibeLogger.CategoryLogger LOG = VibeLogger.forClass(McpClient.class);
    private static final String PROTOCOL_VERSION = "2024-11-05";
    private static final String CLIENT_NAME = "1C Copilot";
    private static final String CLIENT_VERSION = "1.3.0";
    private final IMcpTransport transport;
    private final String serverName;
    private McpServerCapabilities serverCapabilities;
    private List<McpTool> tools = new ArrayList<McpTool>();
    private List<McpResource> resources = new ArrayList<McpResource>();
    private List<McpPrompt> prompts = new ArrayList<McpPrompt>();
    private final Gson gson = new Gson();
    private Consumer<List<McpTool>> toolsChangedListener;

    public McpClient(String serverName, IMcpTransport transport) {
        this.serverName = serverName;
        this.transport = transport;
        transport.setNotificationHandler(this::handleNotification);
    }

    public CompletableFuture<Void> initialize() {
        return this.transport.send(this.createRequest("initialize", Map.of("protocolVersion", PROTOCOL_VERSION, "capabilities", Map.of(), "clientInfo", Map.of("name", CLIENT_NAME, "version", CLIENT_VERSION)))).thenCompose(response -> {
            this.serverCapabilities = McpServerCapabilities.fromInitializeResult(response.getResult());
            LOG.info("MCP server '%s' initialized. Server capabilities: tools=%b, resources=%b, prompts=%b", this.serverName, this.serverCapabilities.supportsTools(), this.serverCapabilities.supportsResources(), this.serverCapabilities.supportsPrompts());
            this.sendNotification("notifications/initialized", null);
            return this.discoverCapabilities();
        });
    }

    private void handleNotification(McpMessage notification) {
        String method = notification.getMethod();
        LOG.debug("Received notification: %s", method);
        if ("notifications/tools/list_changed".equals(method)) {
            LOG.info("Tools list changed notification from %s, refreshing...", this.serverName);
            ((CompletableFuture)this.listTools().thenAccept(newTools -> {
                this.tools = newTools;
                LOG.info("Refreshed tools for %s: %d tools", this.serverName, newTools.size());
                if (this.toolsChangedListener != null) {
                    this.toolsChangedListener.accept((List<McpTool>)newTools);
                }
            })).exceptionally(e -> {
                LOG.warn("Failed to refresh tools for %s: %s", this.serverName, e.getMessage());
                return null;
            });
        } else if ("notifications/resources/list_changed".equals(method)) {
            LOG.info("Resources list changed notification from %s, refreshing...", this.serverName);
            ((CompletableFuture)this.listResources().thenAccept(newResources -> {
                this.resources = newResources;
                LOG.info("Refreshed resources for %s: %d resources", this.serverName, newResources.size());
            })).exceptionally(e -> {
                LOG.warn("Failed to refresh resources for %s: %s", this.serverName, e.getMessage());
                return null;
            });
        } else if ("notifications/prompts/list_changed".equals(method)) {
            LOG.info("Prompts list changed notification from %s, refreshing...", this.serverName);
            ((CompletableFuture)this.listPrompts().thenAccept(newPrompts -> {
                this.prompts = newPrompts;
                LOG.info("Refreshed prompts for %s: %d prompts", this.serverName, newPrompts.size());
            })).exceptionally(e -> {
                LOG.warn("Failed to refresh prompts for %s: %s", this.serverName, e.getMessage());
                return null;
            });
        }
    }

    public void setToolsChangedListener(Consumer<List<McpTool>> listener) {
        this.toolsChangedListener = listener;
    }

    private CompletableFuture<Void> discoverCapabilities() {
        ArrayList<CompletionStage> futures = new ArrayList<CompletionStage>();
        if (this.serverCapabilities.supportsTools()) {
            futures.add(this.listTools().thenAccept(t -> {
                List list = this.tools = t;
            }));
        }
        if (this.serverCapabilities.supportsResources()) {
            futures.add(this.listResources().thenAccept(r -> {
                List list = this.resources = r;
            }));
        }
        if (this.serverCapabilities.supportsPrompts()) {
            futures.add(this.listPrompts().thenAccept(p -> {
                List list = this.prompts = p;
            }));
        }
        return CompletableFuture.allOf(futures.toArray(new CompletableFuture[0]));
    }

    public CompletableFuture<List<McpTool>> listTools() {
        return this.transport.send(this.createRequest("tools/list", null)).thenApply(response -> this.parseToolList(response.getResult()));
    }

    public CompletableFuture<McpToolResult> callTool(String toolName, Map<String, Object> arguments) {
        return this.transport.send(this.createRequest("tools/call", Map.of("name", toolName, "arguments", arguments != null ? arguments : Map.of()))).thenApply(response -> this.parseToolResult(response.getResult()));
    }

    public CompletableFuture<List<McpResource>> listResources() {
        return this.transport.send(this.createRequest("resources/list", null)).thenApply(response -> this.parseResourceList(response.getResult()));
    }

    public CompletableFuture<McpResourceContent> readResource(String uri) {
        return this.transport.send(this.createRequest("resources/read", Map.of("uri", uri))).thenApply(response -> this.parseResourceContent(response.getResult()));
    }

    public CompletableFuture<List<McpPrompt>> listPrompts() {
        return this.transport.send(this.createRequest("prompts/list", null)).thenApply(response -> this.parsePromptList(response.getResult()));
    }

    public CompletableFuture<McpPromptResult> getPrompt(String name, Map<String, String> arguments) {
        return this.transport.send(this.createRequest("prompts/get", Map.of("name", name, "arguments", arguments != null ? arguments : Map.of()))).thenApply(response -> this.parsePromptResult(response.getResult()));
    }

    public String getServerName() {
        return this.serverName;
    }

    public List<McpTool> getTools() {
        return Collections.unmodifiableList(this.tools);
    }

    public List<McpResource> getResources() {
        return Collections.unmodifiableList(this.resources);
    }

    public List<McpPrompt> getPrompts() {
        return Collections.unmodifiableList(this.prompts);
    }

    public boolean isConnected() {
        return this.transport.isConnected();
    }

    public McpServerCapabilities getCapabilities() {
        return this.serverCapabilities;
    }

    @Override
    public void close() {
        try {
            this.transport.close();
        }
        catch (Exception e) {
            LOG.warn("Error closing MCP transport", e);
        }
    }

    private McpMessage createRequest(String method, Object params) {
        McpMessage msg = new McpMessage();
        msg.setMethod(method);
        msg.setParams(params);
        return msg;
    }

    private void sendNotification(String method, Object params) {
        McpMessage msg = new McpMessage();
        msg.setMethod(method);
        msg.setParams(params);
        this.transport.sendNotification(msg).exceptionally(e -> {
            LOG.warn("Failed to send notification %s: %s", method, e.getMessage());
            return null;
        });
    }

    private JsonObject toJsonObject(Object result) {
        if (result == null) {
            return null;
        }
        if (result instanceof JsonObject) {
            return (JsonObject)result;
        }
        JsonElement element = this.gson.toJsonTree(result);
        if (element.isJsonObject()) {
            return element.getAsJsonObject();
        }
        return null;
    }

    private List<McpTool> parseToolList(Object result) {
        ArrayList<McpTool> tools = new ArrayList<McpTool>();
        JsonObject obj = this.toJsonObject(result);
        if (obj != null && obj.has("tools")) {
            JsonArray arr = obj.getAsJsonArray("tools");
            for (JsonElement elem : arr) {
                tools.add((McpTool)this.gson.fromJson(elem, McpTool.class));
            }
            LOG.debug("Parsed %d tools from server %s", tools.size(), this.serverName);
        }
        return tools;
    }

    private McpToolResult parseToolResult(Object result) {
        JsonObject obj = this.toJsonObject(result);
        if (obj != null) {
            return (McpToolResult)this.gson.fromJson((JsonElement)obj, McpToolResult.class);
        }
        return new McpToolResult();
    }

    private List<McpResource> parseResourceList(Object result) {
        ArrayList<McpResource> resources = new ArrayList<McpResource>();
        JsonObject obj = this.toJsonObject(result);
        if (obj != null && obj.has("resources")) {
            JsonArray arr = obj.getAsJsonArray("resources");
            for (JsonElement elem : arr) {
                resources.add((McpResource)this.gson.fromJson(elem, McpResource.class));
            }
            LOG.debug("Parsed %d resources from server %s", resources.size(), this.serverName);
        }
        return resources;
    }

    private McpResourceContent parseResourceContent(Object result) {
        JsonObject obj = this.toJsonObject(result);
        if (obj != null) {
            return (McpResourceContent)this.gson.fromJson((JsonElement)obj, McpResourceContent.class);
        }
        return new McpResourceContent();
    }

    private List<McpPrompt> parsePromptList(Object result) {
        ArrayList<McpPrompt> prompts = new ArrayList<McpPrompt>();
        JsonObject obj = this.toJsonObject(result);
        if (obj != null && obj.has("prompts")) {
            JsonArray arr = obj.getAsJsonArray("prompts");
            for (JsonElement elem : arr) {
                prompts.add((McpPrompt)this.gson.fromJson(elem, McpPrompt.class));
            }
            LOG.debug("Parsed %d prompts from server %s", prompts.size(), this.serverName);
        }
        return prompts;
    }

    private McpPromptResult parsePromptResult(Object result) {
        JsonObject obj = this.toJsonObject(result);
        if (obj != null) {
            return (McpPromptResult)this.gson.fromJson((JsonElement)obj, McpPromptResult.class);
        }
        return new McpPromptResult();
    }
}

