/*
 * Decompiled with CFR 0.152.
 */
package com.example.vibe.core.backend;

import com.example.vibe.core.backend.BackendConfig;
import com.example.vibe.core.backend.BackendException;
import com.example.vibe.core.backend.RegistrationResult;
import com.example.vibe.core.backend.SignupStartResult;
import com.example.vibe.core.backend.UsageInfo;
import com.example.vibe.core.logging.VibeLogger;
import com.example.vibe.core.settings.SecureStorageUtil;
import com.google.gson.Gson;
import com.google.gson.JsonElement;
import com.google.gson.JsonObject;
import com.google.gson.JsonParser;
import java.net.URI;
import java.net.http.HttpClient;
import java.net.http.HttpRequest;
import java.net.http.HttpResponse;
import java.time.Duration;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.CopyOnWriteArrayList;

public class BackendService {
    private static final VibeLogger.CategoryLogger LOG = VibeLogger.forClass(BackendService.class);
    private static final String SECURE_KEY_API_KEY = "backend.apiKey";
    private static final String SECURE_KEY_USER_ID = "backend.userId";
    private static final String SECURE_KEY_USER_EMAIL = "backend.userEmail";
    private static final String CONTENT_TYPE_JSON = "application/json";
    private static final String HEADER_CONTENT_TYPE = "Content-Type";
    private static final String HEADER_AUTHORIZATION = "Authorization";
    private static final String BEARER_PREFIX = "Bearer ";
    private static final long USAGE_CACHE_TTL = 300000L;
    private static final Duration HTTP_TIMEOUT = Duration.ofSeconds(30L);
    private static volatile BackendService instance;
    private final HttpClient httpClient;
    private final Gson gson;
    private final List<UsageChangeListener> usageListeners = new CopyOnWriteArrayList<UsageChangeListener>();
    private volatile UsageInfo cachedUsage;
    private volatile long lastUsageFetch;

    private BackendService() {
        this.httpClient = HttpClient.newBuilder().connectTimeout(HTTP_TIMEOUT).build();
        this.gson = new Gson();
        LOG.info("BackendService initialized");
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     * Enabled force condition propagation
     * Lifted jumps to return sites
     */
    public static BackendService getInstance() {
        if (instance != null) return instance;
        Class<BackendService> clazz = BackendService.class;
        synchronized (BackendService.class) {
            if (instance != null) return instance;
            instance = new BackendService();
            // ** MonitorExit[var0] (shouldn't be in output)
            return instance;
        }
    }

    public CompletableFuture<SignupStartResult> signupStart(String email, String name) {
        LOG.info("Signup start for user: %s", email);
        JsonObject body = new JsonObject();
        body.addProperty("email", email);
        body.addProperty("name", name);
        HttpRequest request = HttpRequest.newBuilder().uri(URI.create(BackendConfig.AUTH_BASE_URL + "/api/plugin/auth/signup/start")).header(HEADER_CONTENT_TYPE, CONTENT_TYPE_JSON).timeout(HTTP_TIMEOUT).POST(HttpRequest.BodyPublishers.ofString(this.gson.toJson((JsonElement)body))).build();
        return ((CompletableFuture)this.httpClient.sendAsync(request, HttpResponse.BodyHandlers.ofString()).thenApply(this::handleSignupStartResponse)).exceptionally(ex -> {
            LOG.error("Signup start failed", BackendService.unwrapException(ex));
            return SignupStartResult.failure("\u0421\u0435\u0440\u0432\u0438\u0441 \u0440\u0435\u0433\u0438\u0441\u0442\u0440\u0430\u0446\u0438\u0438 \u0432\u0440\u0435\u043c\u0435\u043d\u043d\u043e \u043d\u0435\u0434\u043e\u0441\u0442\u0443\u043f\u0435\u043d", "service_unavailable", 0L);
        });
    }

    public CompletableFuture<RegistrationResult> signupConfirm(String email, String name, String password, String verificationCode) {
        LOG.info("Signup confirm for user: %s", email);
        JsonObject body = new JsonObject();
        body.addProperty("email", email);
        body.addProperty("name", name);
        body.addProperty("password", password);
        body.addProperty("verification_code", verificationCode);
        HttpRequest request = HttpRequest.newBuilder().uri(URI.create(BackendConfig.AUTH_BASE_URL + "/api/plugin/auth/signup/confirm")).header(HEADER_CONTENT_TYPE, CONTENT_TYPE_JSON).timeout(HTTP_TIMEOUT).POST(HttpRequest.BodyPublishers.ofString(this.gson.toJson((JsonElement)body))).build();
        return ((CompletableFuture)this.httpClient.sendAsync(request, HttpResponse.BodyHandlers.ofString()).thenApply(this::handlePluginAuthResponse)).exceptionally(ex -> {
            LOG.error("Signup confirm failed", BackendService.unwrapException(ex));
            return RegistrationResult.failure("\u0421\u0435\u0440\u0432\u0438\u0441 \u0440\u0435\u0433\u0438\u0441\u0442\u0440\u0430\u0446\u0438\u0438 \u0432\u0440\u0435\u043c\u0435\u043d\u043d\u043e \u043d\u0435\u0434\u043e\u0441\u0442\u0443\u043f\u0435\u043d", "service_unavailable", 0L);
        });
    }

    public CompletableFuture<RegistrationResult> login(String email, String password) {
        LOG.info("Login request for user: %s", email);
        JsonObject body = new JsonObject();
        body.addProperty("email", email);
        body.addProperty("password", password);
        HttpRequest request = HttpRequest.newBuilder().uri(URI.create(BackendConfig.AUTH_BASE_URL + "/api/plugin/auth/login")).header(HEADER_CONTENT_TYPE, CONTENT_TYPE_JSON).timeout(HTTP_TIMEOUT).POST(HttpRequest.BodyPublishers.ofString(this.gson.toJson((JsonElement)body))).build();
        return ((CompletableFuture)this.httpClient.sendAsync(request, HttpResponse.BodyHandlers.ofString()).thenApply(this::handlePluginAuthResponse)).exceptionally(ex -> {
            LOG.error("Login failed", BackendService.unwrapException(ex));
            return RegistrationResult.failure("\u0421\u0435\u0440\u0432\u0438\u0441 \u0430\u0432\u0442\u043e\u0440\u0438\u0437\u0430\u0446\u0438\u0438 \u0432\u0440\u0435\u043c\u0435\u043d\u043d\u043e \u043d\u0435\u0434\u043e\u0441\u0442\u0443\u043f\u0435\u043d", "service_unavailable", 0L);
        });
    }

    public CompletableFuture<RegistrationResult> register(String email, String name) {
        LOG.info("Registering user: %s", email);
        JsonObject body = new JsonObject();
        body.addProperty("email", email);
        body.addProperty("name", name);
        HttpRequest request = HttpRequest.newBuilder().uri(URI.create(BackendConfig.BASE_URL + "/api/v1/register")).header(HEADER_CONTENT_TYPE, CONTENT_TYPE_JSON).timeout(HTTP_TIMEOUT).POST(HttpRequest.BodyPublishers.ofString(this.gson.toJson((JsonElement)body))).build();
        return ((CompletableFuture)this.httpClient.sendAsync(request, HttpResponse.BodyHandlers.ofString()).thenApply(response -> this.handleRegistrationResponse((HttpResponse<String>)response))).exceptionally(ex -> {
            LOG.error("Registration failed", BackendService.unwrapException(ex));
            return RegistrationResult.failure(ex.getCause() != null ? ex.getCause().getMessage() : ex.getMessage());
        });
    }

    private RegistrationResult handleRegistrationResponse(HttpResponse<String> response) {
        String responseBody = response.body();
        LOG.debug("Register response: status=%d", response.statusCode());
        if (response.statusCode() == 200 || response.statusCode() == 201) {
            JsonObject json = JsonParser.parseString((String)responseBody).getAsJsonObject();
            String apiKey = BackendService.getJsonString(json, "api_key");
            String userId = BackendService.getJsonString(json, "user_id");
            double maxBudget = BackendService.getJsonDouble(json, "max_budget");
            if (apiKey != null && !apiKey.isEmpty()) {
                this.storeCredentials(apiKey, userId);
                LOG.info("Registration successful for user: %s", userId);
                return RegistrationResult.success(apiKey, userId, maxBudget);
            }
            return RegistrationResult.failure("No API key in response");
        }
        String errorMessage = BackendService.extractErrorMessage(responseBody, response.statusCode());
        LOG.warn("Registration failed: %s", errorMessage);
        return RegistrationResult.failure(errorMessage);
    }

    public CompletableFuture<UsageInfo> getUsage() {
        String userId = this.getUserId();
        String apiKey = this.getApiKey();
        if (userId == null || userId.isEmpty() || apiKey == null || apiKey.isEmpty()) {
            return CompletableFuture.failedFuture(new BackendException("Not configured: missing credentials", 0, "not_configured"));
        }
        LOG.debug("Fetching usage for user: %s", userId);
        String url = BackendConfig.BASE_URL + "/api/v1/usage?user_id=" + userId;
        HttpRequest request = HttpRequest.newBuilder().uri(URI.create(url)).header(HEADER_AUTHORIZATION, BEARER_PREFIX + apiKey).header(HEADER_CONTENT_TYPE, CONTENT_TYPE_JSON).timeout(HTTP_TIMEOUT).GET().build();
        return ((CompletableFuture)this.httpClient.sendAsync(request, HttpResponse.BodyHandlers.ofString()).thenApply(response -> this.handleUsageResponse((HttpResponse<String>)response))).exceptionally(ex -> {
            LOG.error("Failed to fetch usage", BackendService.unwrapException(ex));
            return this.cachedUsage;
        });
    }

    private UsageInfo handleUsageResponse(HttpResponse<String> response) {
        String responseBody = response.body();
        LOG.debug("Usage response: status=%d", response.statusCode());
        if (response.statusCode() == 200) {
            try {
                JsonObject json = JsonParser.parseString((String)responseBody).getAsJsonObject();
                UsageInfo usage = new UsageInfo();
                usage.setSpend(BackendService.getJsonDouble(json, "spend"));
                usage.setMaxBudget(BackendService.getJsonDouble(json, "max_budget"));
                usage.setTotalTokens(BackendService.getJsonLong(json, "total_tokens"));
                usage.setPromptTokens(BackendService.getJsonLong(json, "prompt_tokens"));
                usage.setCompletionTokens(BackendService.getJsonLong(json, "completion_tokens"));
                usage.setBudgetDuration(BackendService.getJsonString(json, "budget_duration"));
                usage.setResetDate(BackendService.getJsonString(json, "budget_reset_at"));
                this.cachedUsage = usage;
                this.lastUsageFetch = System.currentTimeMillis();
                this.notifyUsageListeners(usage);
                LOG.debug("Usage fetched: %s", usage);
                return usage;
            }
            catch (Exception e) {
                LOG.error("Failed to parse usage response", e);
                return this.cachedUsage;
            }
        }
        LOG.warn("Usage request failed: status=%d, body=%s", response.statusCode(), responseBody);
        return this.cachedUsage;
    }

    public CompletableFuture<RegistrationResult> rotateKey() {
        String userId = this.getUserId();
        String apiKey = this.getApiKey();
        if (userId == null || userId.isEmpty() || apiKey == null || apiKey.isEmpty()) {
            return CompletableFuture.failedFuture(new BackendException("Not configured: missing credentials", 0, "not_configured"));
        }
        LOG.info("Rotating API key for user: %s", userId);
        JsonObject body = new JsonObject();
        body.addProperty("user_id", userId);
        HttpRequest request = HttpRequest.newBuilder().uri(URI.create(BackendConfig.BASE_URL + "/api/v1/rotate-key")).header(HEADER_CONTENT_TYPE, CONTENT_TYPE_JSON).header(HEADER_AUTHORIZATION, BEARER_PREFIX + apiKey).timeout(HTTP_TIMEOUT).POST(HttpRequest.BodyPublishers.ofString(this.gson.toJson((JsonElement)body))).build();
        return ((CompletableFuture)this.httpClient.sendAsync(request, HttpResponse.BodyHandlers.ofString()).thenApply(response -> this.handleRotateKeyResponse((HttpResponse<String>)response))).exceptionally(ex -> {
            LOG.error("Key rotation failed", BackendService.unwrapException(ex));
            return RegistrationResult.failure(ex.getCause() != null ? ex.getCause().getMessage() : ex.getMessage());
        });
    }

    private RegistrationResult handleRotateKeyResponse(HttpResponse<String> response) {
        String responseBody = response.body();
        LOG.debug("Rotate key response: status=%d", response.statusCode());
        if (response.statusCode() == 200 || response.statusCode() == 201) {
            JsonObject json = JsonParser.parseString((String)responseBody).getAsJsonObject();
            String newApiKey = BackendService.getJsonString(json, "api_key");
            String userId = this.getUserId();
            double maxBudget = BackendService.getJsonDouble(json, "max_budget");
            if (newApiKey != null && !newApiKey.isEmpty()) {
                this.storeCredentials(newApiKey, userId);
                LOG.info("Key rotation successful");
                return RegistrationResult.success(newApiKey, userId, maxBudget);
            }
            return RegistrationResult.failure("No API key in rotation response");
        }
        String errorMessage = BackendService.extractErrorMessage(responseBody, response.statusCode());
        LOG.warn("Key rotation failed: %s", errorMessage);
        return RegistrationResult.failure(errorMessage);
    }

    private SignupStartResult handleSignupStartResponse(HttpResponse<String> response) {
        String responseBody = response.body();
        LOG.debug("Signup start response: status=%d", response.statusCode());
        JsonObject json = BackendService.parseJsonObject(responseBody);
        if ((response.statusCode() == 200 || response.statusCode() == 201) && BackendService.isUnifiedSuccess(json)) {
            JsonObject data = BackendService.getDataObject(json);
            return SignupStartResult.success(BackendService.getJsonBoolean(data, "verification_required", true), BackendService.getJsonLong(data, "expires_in_seconds"), BackendService.getJsonLong(data, "resend_available_in_seconds"));
        }
        ErrorDetails error = BackendService.extractErrorDetails(json, responseBody, response.statusCode());
        LOG.warn("Signup start failed: %s (%s)", error.message, error.errorCode);
        return SignupStartResult.failure(error.message, error.errorCode, error.retryAfterSeconds);
    }

    private RegistrationResult handlePluginAuthResponse(HttpResponse<String> response) {
        String responseBody = response.body();
        LOG.debug("Plugin auth response: status=%d", response.statusCode());
        JsonObject json = BackendService.parseJsonObject(responseBody);
        if ((response.statusCode() == 200 || response.statusCode() == 201) && BackendService.isUnifiedSuccess(json)) {
            JsonObject data = BackendService.getDataObject(json);
            String apiKey = BackendService.firstNonEmpty(BackendService.getJsonString(data, "backendApiKey"), BackendService.getJsonString(data, "backend_api_key"), BackendService.getJsonString(data, "api_key"));
            String userId = BackendService.firstNonEmpty(BackendService.getJsonString(data, "backendUserId"), BackendService.getJsonString(data, "backend_user_id"), BackendService.getJsonString(data, "user_id"));
            String email = BackendService.getJsonString(data, "email");
            if (apiKey != null && !apiKey.isEmpty() && userId != null && !userId.isEmpty()) {
                this.storeCredentials(apiKey, userId, email);
                return RegistrationResult.success(apiKey, userId, email, 0.0);
            }
            return RegistrationResult.failure("\u041d\u0435 \u0443\u0434\u0430\u043b\u043e\u0441\u044c \u043f\u043e\u043b\u0443\u0447\u0438\u0442\u044c \u043a\u043b\u044e\u0447 \u0434\u043e\u0441\u0442\u0443\u043f\u0430", "service_unavailable", 0L);
        }
        ErrorDetails error = BackendService.extractErrorDetails(json, responseBody, response.statusCode());
        LOG.warn("Plugin auth failed: %s (%s)", error.message, error.errorCode);
        return RegistrationResult.failure(error.message, error.errorCode, error.retryAfterSeconds);
    }

    public void storeCredentials(String apiKey, String userId) {
        this.storeCredentials(apiKey, userId, null);
    }

    public void storeCredentials(String apiKey, String userId, String email) {
        SecureStorageUtil.storeSecurely(SECURE_KEY_API_KEY, apiKey);
        SecureStorageUtil.storeSecurely(SECURE_KEY_USER_ID, userId);
        if (email != null && !email.isEmpty()) {
            SecureStorageUtil.storeSecurely(SECURE_KEY_USER_EMAIL, email);
        }
        LOG.debug("Credentials stored for user: %s", userId);
    }

    public String getApiKey() {
        return SecureStorageUtil.retrieveSecurely(SECURE_KEY_API_KEY, "");
    }

    public String getUserId() {
        return SecureStorageUtil.retrieveSecurely(SECURE_KEY_USER_ID, "");
    }

    public String getUserEmail() {
        return SecureStorageUtil.retrieveSecurely(SECURE_KEY_USER_EMAIL, "");
    }

    public boolean isConfigured() {
        String apiKey = this.getApiKey();
        String userId = this.getUserId();
        return apiKey != null && !apiKey.isEmpty() && userId != null && !userId.isEmpty();
    }

    public void clearCredentials() {
        SecureStorageUtil.removeSecurely(SECURE_KEY_API_KEY);
        SecureStorageUtil.removeSecurely(SECURE_KEY_USER_ID);
        SecureStorageUtil.removeSecurely(SECURE_KEY_USER_EMAIL);
        this.cachedUsage = null;
        this.lastUsageFetch = 0L;
        LOG.info("Credentials cleared");
    }

    public UsageInfo getCachedUsage() {
        return this.cachedUsage;
    }

    public boolean isUsageCacheValid() {
        return this.cachedUsage != null && System.currentTimeMillis() - this.lastUsageFetch < 300000L;
    }

    public void refreshUsage() {
        if (!this.isConfigured()) {
            LOG.debug("Skipping usage refresh: not configured");
            return;
        }
        this.getUsage().thenAccept(usage -> {
            if (usage != null) {
                LOG.debug("Usage refreshed: %s", usage);
            }
        });
    }

    public void addUsageListener(UsageChangeListener listener) {
        if (listener != null) {
            this.usageListeners.add(listener);
        }
    }

    public void removeUsageListener(UsageChangeListener listener) {
        if (listener != null) {
            this.usageListeners.remove(listener);
        }
    }

    private void notifyUsageListeners(UsageInfo usage) {
        for (UsageChangeListener listener : this.usageListeners) {
            try {
                listener.onUsageChanged(usage);
            }
            catch (Exception e) {
                LOG.warn("Error in usage listener", e);
            }
        }
    }

    public void dispose() {
        this.usageListeners.clear();
        this.cachedUsage = null;
        this.lastUsageFetch = 0L;
        LOG.info("BackendService disposed");
    }

    private static String getJsonString(JsonObject json, String key) {
        if (json == null) {
            return null;
        }
        if (json.has(key) && !json.get(key).isJsonNull()) {
            return json.get(key).getAsString();
        }
        return null;
    }

    private static double getJsonDouble(JsonObject json, String key) {
        if (json == null) {
            return 0.0;
        }
        if (json.has(key) && !json.get(key).isJsonNull()) {
            return json.get(key).getAsDouble();
        }
        return 0.0;
    }

    private static long getJsonLong(JsonObject json, String key) {
        if (json == null) {
            return 0L;
        }
        if (json.has(key) && !json.get(key).isJsonNull()) {
            return json.get(key).getAsLong();
        }
        return 0L;
    }

    private static boolean getJsonBoolean(JsonObject json, String key, boolean defaultValue) {
        if (json == null) {
            return defaultValue;
        }
        if (json.has(key) && !json.get(key).isJsonNull()) {
            return json.get(key).getAsBoolean();
        }
        return defaultValue;
    }

    private static JsonObject parseJsonObject(String responseBody) {
        try {
            JsonElement element = JsonParser.parseString((String)responseBody);
            if (element != null && element.isJsonObject()) {
                return element.getAsJsonObject();
            }
        }
        catch (Exception exception) {}
        return new JsonObject();
    }

    private static boolean isUnifiedSuccess(JsonObject json) {
        return json.has("success") && !json.get("success").isJsonNull() && json.get("success").getAsBoolean();
    }

    private static JsonObject getDataObject(JsonObject json) {
        if (json.has("data") && json.get("data").isJsonObject()) {
            return json.get("data").getAsJsonObject();
        }
        return json;
    }

    private static String firstNonEmpty(String ... values) {
        if (values == null) {
            return null;
        }
        String[] stringArray = values;
        int n = values.length;
        int n2 = 0;
        while (n2 < n) {
            String value = stringArray[n2];
            if (value != null && !value.isEmpty()) {
                return value;
            }
            ++n2;
        }
        return null;
    }

    private static ErrorDetails extractErrorDetails(JsonObject json, String responseBody, int statusCode) {
        String errorCode = BackendService.getJsonString(json, "error_code");
        String message = BackendService.getJsonString(json, "message");
        long retryAfterSeconds = BackendService.getJsonLong(json, "retry_after");
        if (message == null || message.isEmpty()) {
            message = BackendService.extractErrorMessage(responseBody, statusCode);
        }
        if (errorCode == null || errorCode.isEmpty()) {
            errorCode = statusCode == 429 ? "rate_limited" : (statusCode >= 500 ? "service_unavailable" : "internal_error");
        }
        return new ErrorDetails(message, errorCode, retryAfterSeconds);
    }

    private static String extractErrorMessage(String responseBody, int statusCode) {
        try {
            JsonObject json = JsonParser.parseString((String)responseBody).getAsJsonObject();
            String error = BackendService.getJsonString(json, "error");
            if (error != null) {
                return error;
            }
            String message = BackendService.getJsonString(json, "message");
            if (message != null) {
                return message;
            }
        }
        catch (Exception exception) {}
        return "HTTP " + statusCode + ": " + responseBody;
    }

    private static Throwable unwrapException(Throwable ex) {
        return ex.getCause() != null ? ex.getCause() : ex;
    }

    private static final class ErrorDetails {
        private final String message;
        private final String errorCode;
        private final long retryAfterSeconds;

        private ErrorDetails(String message, String errorCode, long retryAfterSeconds) {
            this.message = message;
            this.errorCode = errorCode;
            this.retryAfterSeconds = retryAfterSeconds;
        }
    }

    public static interface UsageChangeListener {
        public void onUsageChanged(UsageInfo var1);
    }
}

